import bpy
import bpy.types
import math as m
import time
import os
import random as r
import array
import mathutils
import json
import urllib.request
import urllib.error
from typing import Optional

from mathutils import Vector, Matrix, Euler
import bmesh
import collections
from math import pi
# For Printing
import builtins as __builtin__
import warnings
import contextlib
import sys
import importlib
import gc
import colorsys

##########################################################################
# Global Variables for Grok API Configuration
# These are module-level globals for easy access and modification across functions.
# Set them once at module load; the function will use/read them dynamically.
#  
# Global: Holds the xAI API key (read from grok.txt on first use)
g_grok_key = "xai-"
g_grok_endpoint = "https://api.x.ai/v1/chat/completions"  # Global: API endpoint URL
g_grok_model = "grok-4-fast-reasoning"  # Global: Default model (can be overridden per call)
g_grok_max_tokens = 1024  # Global: Default max tokens for responses
g_grok_temperature = 0.7  # Global: Default temperature for creativity control
g_grok_timeout = 30  # Global: Default request timeout in seconds
g_grok_max_retries = 3  # Global: Default max retries for transient errors
g_grok_log_file = "grok_api.log"  # Global: Log file name (relative to cwd)
g_grok_response_log = "grok_response.log"  # Global: Response log file name
##########################################################################
# Insert your API-Key from OpenAI here.
g_Open_ai_key="INSERT HERE YOUR OPEN AI-API-Key"
g_Open_ai_model=""
##########################################################################
# Global Variables for Hunyuan API Configuration (Persistent, like Winsock globals)
# Enhanced: JSON-configurable load from 'hunyuan_config.json' for batch scripting.
g_hunyuan_api_key = ""  # Global: Holds the API key (lazy-load from env/file)
g_hunyuan_endpoint = "http://localhost:7860"  # Wait, no—for Hunyuan: "https://hf.space/gradio/tencent/Hunyuan3D-2.1/api" if custom
g_hunyuan_space = "tencent/Hunyuan3D-2.1"  # Gradio space name (public default)
g_hunyuan_use_key = False  # Flag: True if local/Cloud mode (sets auth header)
g_hunyuan_log_file = "hunyuan_api.log"  # Log for auth traces
##########################################################################
# Global Variables
Undo_VariableL1 = list()
Undo_VariableR1 = list()
BBN = 1
LIF = 0

# Parameters: lst (list) - Input list to check.
# Usage: Removes the first element of the list if it matches the second element. Returns the modified list.
def Py_Check_First(lst):
    """Remove first list element if equal to the second."""
    num = len(lst)
    if num > 1:
        if lst[0] == lst[1]:
            lst.pop(0)
    return lst
##########################################################################
#
# Parameters: ob (bpy.types.Object or str) - Object or object name to check.
# Usage: Checks if the object is linked from an external file (not for linked duplicates). Prints the library status and returns True if linked, False otherwise.
def Is_Obj_File_linked(ob):
    """Check if an object is linked from an external file."""
    myo = bpy.data.objects[ob.name]
    re = myo.library
    print(f"Library check for {ob.name}: {re}")
    return re is not None
##########################################################################
#
# Parameters: *args (variable arguments) - Values to print; **kwargs - Additional print arguments.
# Usage: Prints text to the Blender Python console. Falls back to system console if no console area is open.
def cprint(*args, **kwargs):
    """Print text to the Blender Python console."""
    # Find console area
    console_area = None
    for area in bpy.context.screen.areas:
        if area.type == 'CONSOLE':
            console_area = area
            break
    if not console_area:
        print("Warning: No CONSOLE area found. Printing to system console instead.")
        __builtin__.print(*args, **kwargs)
        return

    # Prepare text
    s = " ".join([str(arg) for arg in args])
    # Use temp_override for operator execution
    with bpy.context.temp_override(area=console_area, space_data=console_area.spaces.active, region=console_area.regions[-1]):
        for line in s.split("\n"):
            bpy.ops.console.scrollback_append(text=line)
##########################################################################

# Parameters: *args (variable arguments) - Values to print; **kwargs - Additional print arguments.
# Usage: Prints text to both the Blender Python console and system console.
def dprint(*args, **kwargs):
    """Print text to both Blender Python console and system console."""
    cprint(*args, **kwargs)  # to Blender console
    __builtin__.print(*args, **kwargs)  # to system console
##########################################################################

# Parameters: None.
# Usage: Finds the console area and space in the current screen. Returns (area, space) or (None, None) if not found.
def console_get():
    """Find the console area and space, returning None if not found."""
    for area in bpy.context.screen.areas:
        if area.type == 'CONSOLE':
            for space in area.spaces:
                if space.type == 'CONSOLE':
                    return area, space
    return None, None
##########################################################################

# Parameters: text (str) - Text to write to the console.
# Usage: Writes text to the Blender Python console, splitting by newlines. Falls back to system console if no console area is open.
def cwrite(text):
    """Write text to the Blender Python console."""
    # Find console area
    console_area = None
    for area in bpy.context.screen.areas:
        if area.type == 'CONSOLE':
            console_area = area
            break
    if not console_area:
        print(f"Warning: No CONSOLE area found. Writing to system console: {text}")
        return

    # Use temp_override for operator execution
    with bpy.context.temp_override(area=console_area, space_data=console_area.spaces.active, region=console_area.regions[-1]):
        for line in text.split("\n"):
            bpy.ops.console.scrollback_append(text=line, type='OUTPUT')
##########################################################################

# Parameters: None.
# Usage: Refreshes the active view layer to update the scene’s display after changes.
def UpdateView():
    """Refresh the active view layer to reflect changes in the scene."""
    bpy.context.view_layer.update()
##########################################################################

# Parameters: None.
# Usage: Returns the current scene object.
def Get_Scene():
    """Get the current scene."""
    return bpy.context.scene
##########################################################################

# Parameters: None.
# Usage: Alias for Get_Scene(), returns the current scene.
def GS():
    """Alias for Get_Scene."""
    return Get_Scene()
##########################################################################

# Parameters: val (int, optional) - Frame number to set; None to get current frame.
# Usage: Gets or sets the current frame in the scene.
def Current_Frame(val=None):
    """Get or set the current frame."""
    if val is None:
        return GS().frame_current
    GS().frame_current = val
##########################################################################

# Parameters: val (int, optional) - Frame number to set; None to get current frame.
# Usage: Alias for Current_Frame(), gets or sets the current frame.
def CF(val=None):
    """Alias for Current_Frame."""
    return Current_Frame(val)
##########################################################################

# Parameters: ref (str or bpy.types.Object) - Object name or reference to check.
# Usage: Checks if an object exists in the scene. Returns True if it exists, False otherwise.
def Exist_Object(ref):
    """Check if an object exists by name or reference."""
    if is_string(ref):
        return ref in bpy.data.objects
    return ref.name in bpy.data.objects
##########################################################################

# Parameters: oba (str or bpy.types.Object) - Object to rename; name (str) - New name.
# Usage: Renames an object. Returns True if successful, False if name is not a string.
def Rename_Object(oba, name="NewName"):
    """Rename an object by name or reference."""
    obj = Get_Object_Any(oba)
    if is_string(name):
        obj.name = name
        return True
    return False
##########################################################################

# Parameters: mi (int) - Minor version; mv (int) - Major version.
# Usage: Checks if Blender version is at least the specified version. Returns True if current version meets or exceeds, False otherwise.
def If_Blender_Version(mi=80, mv=2):
    """Check if Blender version is at least the specified version."""
    if (mv, mi, 0) > bpy.app.version:
        return False
    return True
##########################################################################

# Parameters: None.
# Usage: Checks if Blender version is 2.80 or higher. Returns True for 2.80+, False otherwise.
def If_Blender28():
    """Check if Blender version is 2.80 or higher."""
    if (2, 80, 0) > bpy.app.version:
        return False
    return True
##########################################################################

# Parameters: xn (str) - Object name.
# Usage: Selects an object by name, compatible with Blender 2.7x and 2.8x+.
def Select_Object_by_Name(xn):
    """Select an object by name, compatible with Blender 2.7x and 2.8x+."""
    if (2, 80, 0) > bpy.app.version:
        bpy.data.objects[xn].select = True
    else:
        bpy.data.objects[xn].select_set(True)
##########################################################################

# Parameters: oba (str or bpy.types.Object) - Object to select.
# Usage: Selects an object by name or reference.
def Select_Object(oba):
    """Select an object by name or reference."""
    obj = Get_Object_Any(oba)
    obj.select_set(True)
##########################################################################

# Parameters: xn (str) - Object name.
# Usage: Checks if an object is selected by name. Returns True if selected, False otherwise.
def Is_Object_Selected_by_name(xn):
    """Check if an object is selected by name."""
    return bpy.data.objects[xn].select_get()
##########################################################################

# Parameters: xn (str or bpy.types.Object) - Object to check.
# Usage: Checks if an object is selected by name or reference. Returns True if selected, False otherwise.
def Is_Object_Selected(xn):
    """Check if an object is selected by name or reference."""
    obj = Get_Object_Any(xn)
    return obj.select_get()
##########################################################################

# Parameters: oba (str or bpy.types.Object) - Object to set as active.
# Usage: Sets an object as the active object. Falls back to the last selected object if the specified object doesn’t exist.
def Set_Active_Object(oba):
    """Set an object as active."""
    obj = Get_Object_Any(oba)
    if not Exist_Object(obj):
        obj = Get_Last_Object()
    Select_Object(obj)
    bpy.context.view_layer.objects.active = obj
##########################################################################

# Parameters: None.
# Usage: Returns the active space data of the current area.
def Get_Active_Window():
    """Get the active space data."""
    return bpy.context.area.spaces.active
##########################################################################

# Parameters: oba (str or bpy.types.Object) - Object to check.
# Usage: Checks if an object exists, returning 1 (True) or 0 (False).
def Exists(oba):
    """Check if an object exists, returning 1 (True) or 0 (False)."""
    obj = Get_Object_Any(oba)
    return 1 if bpy.data.objects.get(obj.name) is not None else 0
##########################################################################

# Parameters: None.
# Usage: Returns the active scene.
def Get_Active_Scene():
    """Get the active scene."""
    return bpy.context.screen.scene
##########################################################################

# Parameters: x (float), y (float), z (float) - Coordinates to set the 3D cursor.
# Usage: Sets the 3D cursor position to the specified coordinates.
def Set_3D_Cursor_Pos(x=0, y=0, z=1):
    """Set the 3D cursor position."""
    bpy.context.scene.cursor.location = (x, y, z)
##########################################################################

# Parameters: None.
# Usage: Returns the 3D cursor position as (x, y, z).
def Get_3D_Cursor_Pos():
    """Get the 3D cursor position as (x, y, z)."""
    cursor = bpy.context.scene.cursor
    return cursor.location.x, cursor.location.y, cursor.location.z
##########################################################################

# Parameters: ref (str or bpy.types.Object) - Object to check.
# Usage: Checks if the provided reference is a string. Returns True if string, False otherwise.
def is_string(ref):
    """Check if a reference is a string."""
    return isinstance(ref, str)
##########################################################################
# Parameters: Engine (str) - Render engine name (e.g., 'CYCLES', 'BLENDER_EEVEE').
# Usage: Sets the scene’s render engine. Validates the engine is available; prints a warning if not.
def Set_Render_Engine(Engine="CYCLES"):
    """Set the render engine for the current scene."""
    available_engines = {'CYCLES', 'BLENDER_EEVEE', 'BLENDER_WORKBENCH'}
    if Engine not in available_engines:
        print(f"Warning: Render engine '{Engine}' may require an add-on or be unavailable.")
    try:
        bpy.context.scene.render.engine = Engine
    except TypeError:
        print(f"Error: Invalid render engine '{Engine}'. Available: {available_engines}")
##########################################################################

# Parameters: None.
# Usage: Sets the render engine to Cycles.
def Set_Render_Cycles():
    """Set the render engine to Cycles."""
    Set_Render_Engine('CYCLES')
##########################################################################

# Parameters: None.
# Usage: Sets the render engine to Eevee.
def Set_Render_Eevee():
    """Set the render engine to Eevee."""
    Set_Render_Engine('BLENDER_EEVEE')
##########################################################################

# Parameters: None.
# Usage: Sets the render engine to Radeon ProRender (requires add-on).
def Set_Render_RPR():
    """Set the render engine to Radeon ProRender."""
    Set_Render_Engine("RPR")
##########################################################################

# Parameters: None.
# Usage: Sets the render engine to POV-Ray (requires add-on).
def Set_Render_POV():
    """Set the render engine to POV-Ray."""
    Set_Render_Engine("POVRAY_RENDER")
##########################################################################

# Parameters: None.
# Usage: Sets the render engine to LuxCore (requires add-on).
def Set_Render_Luxcore():
    """Set the render engine to LuxCore."""
    Set_Render_Engine("LUXCORE")
##########################################################################

# Parameters: use_view (bool) - True to render from the active viewport, False for standard render.
# Usage: Renders a single image and returns the resulting image data.
def Render_Image(use_view=False):
    """Render a single image."""
    try:
        bpy.ops.render.render(use_viewport=use_view)
        return bpy.data.images['Render Result']
    except Exception as e:
        print(f"Error rendering image: {str(e)}")
        return None
##########################################################################

# Parameters: use_view (bool) - True to render from the active viewport, False for standard render.
# Usage: Renders an animation and returns the result.
def Render_Animation(use_view=False):
    """Render an animation."""
    try:
        result = bpy.ops.render.render(animation=True, use_viewport=use_view)
        return result
    except Exception as e:
        print(f"Error rendering animation: {str(e)}")
        return None
##########################################################################

# Parameters: x (int) - Resolution width; y (int) - Resolution height.
# Usage: Sets the render resolution for the scene.
def Render_Set_Resolution(x=3840, y=2160):
    """Set the render resolution."""
    scene = Get_Scene()
    scene.render.resolution_x = x
    scene.render.resolution_y = y
##########################################################################

# Parameters: None.
# Usage: Returns the current render resolution as [width, height].
def Render_Get_Resolution():
    """Get the current render resolution."""
    scene = Get_Scene()
    return [scene.render.resolution_x, scene.render.resolution_y]
##########################################################################

# Parameters: p (int) - Resolution percentage (1-100).
# Usage: Sets the render resolution percentage.
def Render_Set_Percentage(p):
    """Set the render resolution percentage."""
    Get_Scene().render.resolution_percentage = p
##########################################################################

# Parameters: None.
# Usage: Returns the current render resolution percentage.
def Render_Get_Percentage():
    """Get the render resolution percentage."""
    return Get_Scene().render.resolution_percentage
##########################################################################

# Parameters: x (float) - Pixel aspect ratio X; y (float) - Pixel aspect ratio Y.
# Usage: Sets the render pixel aspect ratio.
def Render_Set_Aspect(x, y):
    """Set the render pixel aspect ratio."""
    scene = Get_Scene()
    scene.render.pixel_aspect_x = x
    scene.render.pixel_aspect_y = y
##########################################################################

# Parameters: None.
# Usage: Returns the render pixel aspect ratio as [x, y].
def Render_Get_Aspect():
    """Get the render pixel aspect ratio."""
    scene = Get_Scene()
    return [scene.render.pixel_aspect_x, scene.render.pixel_aspect_y]
##########################################################################

# Parameters: val (int) - Frames per second; ba (float) - FPS base (default 1.0).
# Usage: Sets the scene’s frame rate and base.
def Render_Set_FPS(val, ba=1.0):
    """Set the render frame rate and base."""
    scene = Get_Scene()
    scene.render.fps = val
    scene.render.fps_base = ba
##########################################################################

# Parameters: name (str) - Name for the new object; col (str or bpy.types.Collection, optional) - Collection to link the object to.
# Usage: Creates a new mesh object and links it to a collection (default: active collection).
def Create_Object(name, col=None):
    """Create a new mesh object."""
    m = bpy.data.meshes.new(name)
    o = bpy.data.objects.new(name, m)
    col_ref = None
    if col is None:
        col_ref = bpy.context.view_layer.active_layer_collection.collection
    elif is_string(col):
        if col in bpy.data.collections:
            col_ref = bpy.data.collections[col]
        else:
            col_ref = Create_Collection(col)
    else:
        col_ref = col
    if col_ref:
        col_ref.objects.link(o)
    else:
        print(f"Error: Invalid collection for object {name}.")
    return o
##########################################################################

# Parameters: tocopy (str or bpy.types.Object) - Object to copy; col (str or bpy.types.Collection, optional) - Collection for the new object.
# Usage: Creates a copy of an object, including its data, and links it to a collection.
def Copy_Object(tocopy, col=None):
    """Copy an object and its data."""
    to_copy = Get_Object_Any(tocopy)
    col_ref = None
    if col is None:
        col_ref = bpy.context.view_layer.active_layer_collection.collection
    elif is_string(col):
        if Exists_Collection(col):
            col_ref = Get_Collection(col)
        else:
            col_ref = Create_Collection(col)
    else:
        col_ref = col
    if not col_ref:
        print(f"Error: Invalid collection for copying {to_copy.name}.")
        return None
    new_obj = to_copy.copy()
    if new_obj.data is not None:
        new_obj.data = to_copy.data.copy()
    new_obj.animation_data_clear()
    col_ref.objects.link(new_obj)
    return new_obj
##########################################################################
# Parameters: name (str) - Name for the new collection.
# Usage: Creates a new collection and links it to the scene’s root collection. Returns the new collection.
def NewCollection(name):
    """Create a new collection and link it to the scene."""
    col = bpy.data.collections.new(name)
    bpy.context.scene.collection.children.link(col)
    return col
##########################################################################

# Parameters: col (str or bpy.types.Collection) - Collection to link to; oba (str or bpy.types.Object) - Object to link.
# Usage: Links an object to a specified collection.
def LinkToCollection(col, oba):
    """Link an object to a collection."""
    obj = Get_Object_Any(oba)
    col_ref = Get_Collection(col) if is_string(col) else col
    if col_ref:
        col_ref.objects.link(obj)
    else:
        print(f"Error: Invalid collection {col} for linking object {obj.name}.")
##########################################################################

# Parameters: oba (str or bpy.types.Object) - Object to unlink.
# Usage: Unlinks an object from the scene’s root collection.
def UnlinkObjFromScene(oba):
    """Unlink an object from the scene’s root collection."""
    obj = Get_Object_Any(oba)
    try:
        bpy.context.scene.collection.objects.unlink(obj)
    except Exception as e:
        print(f"Error unlinking object {obj.name} from scene: {str(e)}")
##########################################################################

# Parameters: oba (str or bpy.types.Object) - Object to link.
# Usage: Links an object to the scene’s root collection.
def Link_Obj_to_Scene(oba):
    """Link an object to the scene’s root collection."""
    obj = Get_Object_Any(oba)
    try:
        bpy.context.scene.collection.objects.link(obj)
    except Exception as e:
        print(f"Error linking object {obj.name} to scene: {str(e)}")
##########################################################################

# Parameters: col (str or bpy.types.Collection) - Target collection; oba (str or bpy.types.Object) - Object to move.
# Usage: Moves an object to a specified collection by linking it and unlinking from the scene.
def MoveObjToCollection(col, oba):
    """Move an object to a specified collection."""
    obj = Get_Object_Any(oba)
    LinkToCollection(col, obj)
    UnlinkObjFromScene(obj)
##########################################################################

# Parameters: name (str) - Name for the new collection.
# Usage: Creates a new collection if it doesn’t exist and links it to the scene. Returns the collection or False if it already exists.
def Create_Collection(name):
    """Create a new collection if it doesn’t exist."""
    if not Exists_Collection(name):
        col_ref = bpy.data.collections.new(name)
        bpy.context.scene.collection.children.link(col_ref)
        return col_ref
    return False
##########################################################################

# Parameters: name (str or bpy.types.Collection) - Collection to delete; delete_objects (bool) - True to delete objects in the collection.
# Usage: Deletes a collection and optionally its objects. Returns False if the collection doesn’t exist.
def Delete_Collection(name, delete_objects=False):
    """Delete a collection and optionally its objects."""
    if not Exists_Collection(name):
        return False
    col = Get_Collection(name) if is_string(name) else name
    if delete_objects:
        Deselect_All()
        if col.objects:
            for co in col.objects:
                co.select_set(True)
            delete_selected_objects()
    else:
        Deselect_All()
        if col.objects:
            for co in col.objects:
                bpy.context.scene.collection.objects.link(co)
    bpy.data.collections.remove(col)
    return True
##########################################################################

# Parameters: name (str) - Collection name to remove (default: "Collection 1").
# Usage: Removes a collection and its objects if they have no other users.
def Remove_Collection(name="Collection 1"):
    """Remove a collection and its unreferenced objects."""
    coll = bpy.data.collections.get(name)
    if coll:
        remove_collection_objects = True
        if remove_collection_objects:
            obs = [o for o in coll.objects if o.users == 1]
            while obs:
                bpy.data.objects.remove(obs.pop())
        bpy.data.collections.remove(coll)
    else:
        print(f"Error: Collection {name} not found.")
##########################################################################

# Parameters: col (str or bpy.types.Collection) - Collection to clear.
# Usage: Deletes all objects in the specified collection.
def delete_objects_in_collection(col):
    """Delete all objects in a collection."""
    col_ref = Get_Collection(col) if is_string(col) else col
    if not col_ref:
        print(f"Error: Invalid collection {col}.")
        return
    Deselect_All()
    for co in col_ref.objects:
        co.select_set(True)
    delete_selected_objects()
##########################################################################

# Parameters: col (str or bpy.types.Collection) - Collection to delete recursively.
# Usage: Deletes a collection and its child collections, including all objects.
def delete_hierarchy(col):
    """Delete a collection and its child collections recursively."""
    col_ref = Get_Collection(col) if is_string(col) else col
    if not col_ref:
        print(f"Error: Invalid collection {col}.")
        return
    for co in col_ref.children:
        if isinstance(co, bpy.types.Collection):
            delete_hierarchy(co)
    Deselect_All()
    delete_objects_in_collection(col_ref)
    Delete_Collection(col_ref)
##########################################################################

# Parameters: col (str or bpy.types.Collection) - Collection to duplicate.
# Usage: Duplicates a collection and its objects, returning the new collection.
def duplicate_collection(col):
    """Duplicate a collection and its objects."""
    col_ref = Get_Collection(col) if is_string(col) else col
    if not col_ref:
        print(f"Error: Invalid collection {col}.")
        return None
    new_name = "Copy of " + col_ref.name
    new_col = Create_Collection(new_name)
    to_copy = Get_Object_Anys_from_collection(col_ref.name)
    for o in to_copy:
        copy_object(o, new_col)
    return Get_Collection(new_name)
##########################################################################

# Parameters: col (str or bpy.types.Collection) - Collection to query.
# Usage: Returns a list of objects in the specified collection.
def Get_Object_Anys_from_collection(col):
    """Get all objects in a collection."""
    if is_string(col):
        if col in bpy.data.collections:
            return bpy.data.collections[col].objects
        else:
            print(f"Error: Collection {col} not found.")
            return []
    return col.objects
##########################################################################

# Parameters: ref (str or bpy.types.Collection, optional) - Collection name or reference; None for active collection.
# Usage: Returns the specified collection or the active collection if None.
def Get_Collection(ref=None):
    """Get a collection by name or reference."""
    if ref is None:
        return bpy.context.view_layer.active_layer_collection.collection
    if ref in bpy.data.collections:
        return bpy.data.collections[ref]
    print(f"Error: Collection {ref} not found.")
    return False
##########################################################################

# Parameters: None.
# Usage: Returns the active collection in the view layer.
def Collection_Get_Active():
    """Get the active collection."""
    return bpy.context.view_layer.active_layer_collection.collection
##########################################################################

# Parameters: ref (str or bpy.types.Collection) - Collection to set as active.
# Usage: Sets the specified collection as the active collection in the view layer.
def Collection_Set_Active(ref):
    """Set a collection as active."""
    col_ref = Get_Collection(ref) if is_string(ref) else ref
    if col_ref.name in bpy.data.collections:
        bpy.context.view_layer.active_layer_collection = bpy.context.view_layer.layer_collection.children[col_ref.name]
    else:
        print(f"Error: Collection {ref} not found.")
        return False
##########################################################################

# Parameters: None.
# Usage: Returns all collections in the scene.
def Get_all_Collections():
    """Get all collections in the scene."""
    return bpy.data.collections
##########################################################################

# Parameters: ref (str or bpy.types.Object) - Object to link; col (str or bpy.types.Collection) - Target collection.
# Usage: Links a single object to a collection.
def Link_Object_to_Collection(ref, col):
    """Link a single object to a collection."""
    obj_ref = Get_Object_Any(ref) if is_string(ref) else ref
    if is_string(col):
        if col in bpy.data.collections:
            bpy.data.collections[col].objects.link(obj_ref)
        else:
            print(f"Error: Collection {col} not found.")
    else:
        col.objects.link(obj_ref)
##########################################################################

# Parameters: ref (list) - List of objects to link; col (str or bpy.types.Collection) - Target collection.
# Usage: Links multiple objects to a collection.
def Link_Objects_to_Collection(ref, col):
    """Link multiple objects to a collection."""
    col_ref = Get_Collection(col) if is_string(col) else col
    if not col_ref:
        print(f"Error: Invalid collection {col}.")
        return
    for o in ref:
        col_ref.objects.link(o)
##########################################################################

# Parameters: ref (str or bpy.types.Object) - Object to unlink; col (str or bpy.types.Collection) - Collection to unlink from.
# Usage: Unlinks a single object from a collection.
def Unlink_Object_from_Collection(ref, col):
    """Unlink an object from a collection."""
    obj_ref = Get_Object_Any(ref) if is_string(ref) else ref
    col_ref = Get_Collection(col) if is_string(col) else col
    if not col_ref:
        print(f"Error: Invalid collection {col}.")
        return
    try:
        col_ref.objects.unlink(obj_ref)
    except Exception as e:
        print(f"Error unlinking object {obj_ref.name} from collection {col_ref.name}: {str(e)}")
##########################################################################

# Parameters: lis (list) - List of objects to unlink; col (str or bpy.types.Collection) - Collection to unlink from.
# Usage: Unlinks multiple objects from a collection.
def Unlink_Objects_from_Collection(lis, col):
    """Unlink multiple objects from a collection."""
    col_ref = Get_Collection(col) if is_string(col) else col
    if not col_ref:
        print(f"Error: Invalid collection {col}.")
        return
    for o in lis:
        try:
            col_ref.objects.unlink(o)
        except Exception as e:
            print(f"Error unlinking object {o.name} from collection {col_ref.name}: {str(e)}")
##########################################################################

# Parameters: ref (str or bpy.types.Object) - Object to move; col (str or bpy.types.Collection) - Target collection.
# Usage: Moves an object to a collection by unlinking from all current collections and linking to the target.
def Move_Object_to_Collection(ref, col):
    """Move an object to a collection."""
    obj_ref = Get_Object_Any(ref) if is_string(ref) else ref
    col_ref = Get_Collection(col) if is_string(col) else col
    if not col_ref:
        print(f"Error: Invalid collection {col}.")
        return
    cols = obj_ref.users_collection
    for c in cols:
        try:
            c.objects.unlink(obj_ref)
        except Exception as e:
            print(f"Error unlinking object {obj_ref.name} from collection {c.name}: {str(e)}")
    Link_Object_to_Collection(obj_ref, col_ref)
##########################################################################

# Parameters: ref (list) - List of objects to move; col (str or bpy.types.Collection) - Target collection.
# Usage: Moves multiple objects to a collection by unlinking from current collections and linking to the target.
def Move_Objects_to_Collection(ref, col):
    """Move multiple objects to a collection."""
    col_ref = Get_Collection(col) if is_string(col) else col
    if not col_ref:
        print(f"Error: Invalid collection {col}.")
        return
    for o in ref:
        for c in o.users_collection:
            try:
                c.objects.unlink(o)
            except Exception as e:
                print(f"Error unlinking object {o.name} from collection {c.name}: {str(e)}")
        Link_Object_to_Collection(o, col_ref)
##########################################################################

# Parameters: ref (str or bpy.types.Object) - Object to query.
# Usage: Returns the first collection an object belongs to.
def Get_Object_Any_collection(ref):
    """Get the first collection an object belongs to."""
    obj_ref = Get_Object_Any(ref) if is_string(ref) else ref
    return obj_ref.users_collection[0] if obj_ref.users_collection else None
##########################################################################

# Parameters: ref (str or bpy.types.Object) - Object to query.
# Usage: Returns all collections an object belongs to.
def Get_Object_Any_collections(ref):
    """Get all collections an object belongs to."""
    obj_ref = Get_Object_Any(ref) if is_string(ref) else ref
    return obj_ref.users_collection
##########################################################################

# Parameters: col (str or bpy.types.Collection) - Collection to check.
# Usage: Checks if a collection exists by name or reference. Returns True if it exists, False otherwise.
def Exists_Collection(col):
    """Check if a collection exists."""
    if is_string(col):
        return col in bpy.data.collections
    return col.name in bpy.data.collections
##########################################################################

# Parameters: ac (int, optional) - Legacy: 0=disable, 1=enable (deprecated); disable (bool) - True to disable rendering, False to enable.
# Usage: Sets render visibility for selected collections in the Outliner. Use 'disable' parameter; 'ac' is deprecated.
def Outliner_Set_Render_Selected(ac=None, disable=True):
    """Set render visibility for selected collections in the Outliner."""
    if ac is not None:
        warnings.warn(
            "The 'ac' parameter is deprecated. Use 'disable' instead.",
            DeprecationWarning,
            stacklevel=2
        )
        disable = (ac == 0)
    selected_collections = []
    for collection in bpy.data.collections:
        if collection.is_evaluated and collection.name in bpy.context.view_layer.layer_collection.children:
            layer_coll = bpy.context.view_layer.layer_collection.children[collection.name]
            if layer_coll.is_selected:
                selected_collections.append(collection)
    if not selected_collections:
        print("Warning: No collections selected in the Outliner.")
        return
    for collection in selected_collections:
        collection.hide_render = disable
        print(f"{'Disabled' if disable else 'Enabled'} render for collection: {collection.name}")
    for area in bpy.context.screen.areas:
        if area.type == 'OUTLINER':
            area.tag_redraw()
            break
##########################################################################

# Parameters: ac (int, optional) - Legacy: 0=disable, 1=enable (deprecated); disable (bool) - True to disable rendering, False to enable.
# Usage: Sets render visibility for all collections in the scene. Use 'disable' parameter; 'ac' is deprecated.
def Outliner_Set_Render_All(ac=None, disable=True):
    """Set render visibility for all collections in the scene."""
    if ac is not None:
        warnings.warn(
            "The 'ac' parameter is deprecated. Use 'disable' instead.",
            DeprecationWarning,
            stacklevel=2
        )
        disable = (ac == 0)
    for collection in bpy.data.collections:
        collection.hide_render = disable
    for area in bpy.context.screen.areas:
        if area.type == 'OUTLINER':
            area.tag_redraw()
            break
##########################################################################

# Parameters: ac (int, optional) - Legacy: 0=exclude, 1=include (deprecated); exclude (bool) - True to exclude, False to include.
# Usage: Excludes selected collections from the view layer. Use 'exclude' parameter; 'ac' is deprecated.
def Outliner_Deactivate_Selected(ac=None, exclude=True):
    """Exclude selected collections from the view layer."""
    if ac is not None:
        warnings.warn(
            "The 'ac' parameter is deprecated. Use 'exclude' instead.",
            DeprecationWarning,
            stacklevel=2
        )
        exclude = (ac == 0)
    selected_collections = []
    for collection in bpy.data.collections:
        if collection.is_evaluated and collection.name in bpy.context.view_layer.layer_collection.children:
            layer_coll = bpy.context.view_layer.layer_collection.children[collection.name]
            if layer_coll.is_selected:
                selected_collections.append(layer_coll)
    if not selected_collections:
        print("Warning: No collections selected in the Outliner.")
        return
    for layer_coll in selected_collections:
        layer_coll.exclude = exclude
        print(f"{'Excluded' if exclude else 'Included'} collection: {layer_coll.collection.name}")
    for area in bpy.context.screen.areas:
        if area.type == 'OUTLINER':
            area.tag_redraw()
            break
##########################################################################

# Parameters: ac (int, optional) - Legacy: 0=exclude, 1=include (deprecated); exclude (bool) - True to exclude, False to include.
# Usage: Excludes all collections from the view layer. Use 'exclude' parameter; 'ac' is deprecated.
def Outliner_Deactivate_All(ac=None, exclude=True):
    """Exclude all collections from the view layer."""
    if ac is not None:
        warnings.warn(
            "The 'ac' parameter is deprecated. Use 'exclude' instead.",
            DeprecationWarning,
            stacklevel=2
        )
        exclude = (ac == 0)
    for collection in bpy.data.collections:
        if collection.name in bpy.context.view_layer.layer_collection.children:
            layer_coll = bpy.context.view_layer.layer_collection.children[collection.name]
            layer_coll.exclude = exclude
    for area in bpy.context.screen.areas:
        if area.type == 'OUTLINER':
            area.tag_redraw()
            break
##########################################################################

# Parameters: ac (str) - Action: 'TOGGLE', 'SELECT', 'DESELECT', 'INVERT'.
# Usage: Performs a selection action on all Outliner elements.
def Outliner_Select_all(ac='TOGGLE'):
    """Select or deselect all elements in the Outliner."""
    outliner_area = None
    for area in bpy.context.screen.areas:
        if area.type == 'OUTLINER':
            outliner_area = area
            break
    if not outliner_area:
        print("Warning: No OUTLINER area found.")
        return
    try:
        with bpy.context.temp_override(area=outliner_area):
            bpy.ops.outliner.select_all(action=ac)
        outliner_area.tag_redraw()
    except Exception as e:
        print(f"Error performing Outliner selection: {str(e)}")
##########################################################################

# Parameters: None.
# Usage: Shows the active object or collection in the Outliner.
def Outliner_Show_Active():
    """Show the active object or collection in the Outliner."""
    outliner_area = None
    for area in bpy.context.screen.areas:
        if area.type == 'OUTLINER':
            outliner_area = area
            break
    if not outliner_area:
        print("Warning: No OUTLINER area found.")
        return
    try:
        with bpy.context.temp_override(area=outliner_area):
            bpy.ops.outliner.show_active()
        outliner_area.tag_redraw()
    except Exception as e:
        print(f"Error showing active in Outliner: {str(e)}")
##########################################################################

# Parameters: None.
# Usage: Expands the Outliner hierarchy to show all entries.
def Outliner_Show_Hierarchy():
    """Expand the Outliner hierarchy."""
    outliner_area = None
    for area in bpy.context.screen.areas:
        if area.type == 'OUTLINER':
            outliner_area = area
            break
    if not outliner_area:
        print("Warning: No OUTLINER area found.")
        return
    try:
        with bpy.context.temp_override(area=outliner_area):
            bpy.ops.outliner.show_hierarchy()
        outliner_area.tag_redraw()
    except Exception as e:
        print(f"Error showing Outliner hierarchy: {str(e)}")
##########################################################################

# Parameters: ac (int, optional) - Legacy: 1=show, 0=hide (deprecated); show (bool) - True to show, False to hide.
# Usage: Shows or hides all collections in the Outliner. Use 'show' parameter; 'ac' is deprecated.
def Outliner_UnHide_All(ac=None, show=True):
    """Show or hide all collections in the Outliner."""
    if ac is not None:
        warnings.warn(
            "The 'ac' parameter is deprecated. Use 'show' instead.",
            DeprecationWarning,
            stacklevel=2
        )
        show = (ac == 1)
    for collection in bpy.data.collections:
        if collection.name in bpy.context.view_layer.layer_collection.children:
            layer_coll = bpy.context.view_layer.layer_collection.children[collection.name]
            layer_coll.hide_viewport = not show
    for area in bpy.context.screen.areas:
        if area.type == 'OUTLINER':
            area.tag_redraw()
            break
##########################################################################

# Parameters: ac (int, optional) - Legacy: 1=show, 0=hide (deprecated); show (bool) - True to show, False to hide.
# Usage: Shows or hides selected collections in the Outliner. Use 'show' parameter; 'ac' is deprecated.
def Outliner_UnHide_Selected(ac=None, show=True):
    """Show or hide selected collections in the Outliner."""
    if ac is not None:
        warnings.warn(
            "The 'ac' parameter is deprecated. Use 'show' instead.",
            DeprecationWarning,
            stacklevel=2
        )
        show = (ac == 1)
    selected_collections = []
    for collection in bpy.data.collections:
        if collection.is_evaluated and collection.name in bpy.context.view_layer.layer_collection.children:
            layer_coll = bpy.context.view_layer.layer_collection.children[collection.name]
            if layer_coll.is_selected:
                selected_collections.append(layer_coll)
    if not selected_collections:
        print("Warning: No collections selected in the Outliner.")
        return
    for layer_coll in selected_collections:
        layer_coll.hide_viewport = not show
    for area in bpy.context.screen.areas:
        if area.type == 'OUTLINER':
            area.tag_redraw()
            break
##########################################################################

# Parameters: None.
# Usage: Expands all Outliner entries one level deep.
def Outliner_Show_One_Level():
    """Expand all Outliner entries one level deep."""
    outliner_area = None
    for area in bpy.context.screen.areas:
        if area.type == 'OUTLINER':
            outliner_area = area
            break
    if not outliner_area:
        print("Warning: No OUTLINER area found.")
        return
    try:
        with bpy.context.temp_override(area=outliner_area):
            bpy.ops.outliner.show_one_level()
        outliner_area.tag_redraw()
    except Exception as e:
        print(f"Error expanding Outliner one level: {str(e)}")
##########################################################################

# Parameters: ca (bool) - True to hide only selected collections, False to hide all but selected.
# Usage: Isolates selected collections in the Outliner by hiding others.
def Outliner_Isolate_Selected(ca=False):
    """Isolate selected collections in the Outliner."""
    outliner_area = None
    for area in bpy.context.screen.areas:
        if area.type == 'OUTLINER':
            outliner_area = area
            break
    if not outliner_area:
        print("Warning: No OUTLINER area found.")
        return
    try:
        with bpy.context.temp_override(area=outliner_area):
            bpy.ops.outliner.collection_isolate(extend=ca)
        outliner_area.tag_redraw()
    except Exception as e:
        print(f"Error isolating collections in Outliner: {str(e)}")
##########################################################################

# Parameters: state (int) - 0=expand all, 1=expand lower, 2=collapse all, 3=expand all.
# Usage: Controls the Outliner’s hierarchy expansion/collapse state.
def Collapse_Outliner(state):
    """Control Outliner hierarchy expansion/collapse."""
    outliner_area = None
    for area in bpy.context.screen.areas:
        if area.type == 'OUTLINER':
            outliner_area = area
            break
    if not outliner_area:
        print("Warning: No OUTLINER area found.")
        return
    try:
        with bpy.context.temp_override(area=outliner_area):
            bpy.ops.outliner.show_hierarchy()
            for _ in range(state):
                bpy.ops.outliner.expanded_toggle()
        outliner_area.tag_redraw()
    except Exception as e:
        print(f"Error collapsing Outliner: {str(e)}")
##########################################################################

# Parameters: None.
# Usage: Returns the Outliner context area.
def Get_Outliner_Context():
    """Get the Outliner context area."""
    for area in bpy.context.screen.areas:
        if area.type == 'OUTLINER':
            return area
    print("Warning: No OUTLINER area found.")
    return None
##########################################################################

# Parameters: None.
# Usage: Returns an override dictionary for the Outliner area.
def Get_Outliner_Override():
    """Get an override dictionary for the Outliner area."""
    outliner_area = Get_Outliner_Context()
    if outliner_area:
        override = bpy.context.copy()
        override['area'] = outliner_area
        return override
    return {}
##########################################################################
# Parameters: None.
# Usage: Selects all mesh objects in the scene.
def select_all_meshes():
    """Select all mesh objects."""
    bpy.ops.object.select_by_type(type='MESH')
##########################################################################

# Parameters: None.
# Usage: Selects all curve objects in the scene.
def select_all_curves():
    """Select all curve objects."""
    bpy.ops.object.select_by_type(type='CURVE')
##########################################################################

# Parameters: None.
# Usage: Selects all surface objects in the scene.
def select_all_surfaces():
    """Select all surface objects."""
    bpy.ops.object.select_by_type(type='SURFACE')
##########################################################################

# Parameters: None.
# Usage: Selects all meta objects in the scene.
def select_all_metas():
    """Select all meta objects."""
    bpy.ops.object.select_by_type(type='META')
##########################################################################

# Parameters: None.
# Usage: Selects all text objects in the scene.
def select_all_text():
    """Select all text objects."""
    bpy.ops.object.select_by_type(type='FONT')
##########################################################################

# Parameters: None.
# Usage: Selects all hair objects in the scene.
def select_all_hair():
    """Select all hair objects."""
    bpy.ops.object.select_by_type(type='HAIR')
##########################################################################

# Parameters: None.
# Usage: Selects all point cloud objects in the scene.
def select_all_point_clouds():
    """Select all point cloud objects."""
    bpy.ops.object.select_by_type(type='POINTCLOUD')
##########################################################################

# Parameters: None.
# Usage: Selects all volume objects in the scene.
def select_all_volumes():
    """Select all volume objects."""
    bpy.ops.object.select_by_type(type='VOLUME')
##########################################################################

# Parameters: None.
# Usage: Selects all armature objects in the scene.
def select_all_armatures():
    """Select all armature objects."""
    bpy.ops.object.select_by_type(type='ARMATURE')
##########################################################################

# Parameters: None.
# Usage: Selects all lattice objects in the scene.
def select_all_lattices():
    """Select all lattice objects."""
    bpy.ops.object.select_by_type(type='LATTICE')
##########################################################################

# Parameters: None.
# Usage: Selects all empty objects in the scene.
def select_all_empties():
    """Select all empty objects."""
    bpy.ops.object.select_by_type(type='EMPTY')
##########################################################################

# Parameters: None.
# Usage: Selects all grease pencil objects in the scene.
def select_all_greace_pencils():
    """Select all grease pencil objects."""
    bpy.ops.object.select_by_type(type='GPENCIL')
##########################################################################

# Parameters: None.
# Usage: Selects all camera objects in the scene.
def select_all_cameras():
    """Select all camera objects."""
    bpy.ops.object.select_by_type(type='CAMERA')
##########################################################################

# Parameters: None.
# Usage: Selects all speaker objects in the scene.
def select_all_speakers():
    """Select all speaker objects."""
    bpy.ops.object.select_by_type(type='SPEAKER')
##########################################################################

# Parameters: None.
# Usage: Selects all light probe objects in the scene.
def select_all_light_probes():
    """Select all light probe objects."""
    bpy.ops.object.select_by_type(type='LIGHT_PROBE')
##########################################################################

# Parameters: None.
# Usage: Inverts the current selection in the scene.
def invert_selection():
    """Invert the current selection."""
    bpy.ops.object.select_all(action='INVERT')
##########################################################################

# Parameters: ref (str or bpy.types.Object, optional) - Object to hide; None for selected object.
# Usage: Hides an object in the viewport. If no reference, hides the selected object.
def Hide_Object(ref=None):
    """Hide an object in the viewport."""
    if ref is not None:
        obj = Get_Object_Any(ref) if is_string(ref) else ref
        obj.hide_set(True)
    else:
        obj = Get_Active_Object()
        if obj:
            obj.hide_set(True)
        else:
            print("Warning: No object selected to hide.")
##########################################################################

# Parameters: ref (str or bpy.types.Object, optional) - Object to show; None for selected object.
# Usage: Shows an object in the viewport. If no reference, shows the selected object.
def Show_Object(ref=None):
    """Show an object in the viewport."""
    if ref is not None:
        obj = Get_Object_Any(ref) if is_string(ref) else ref
        obj.hide_set(False)
    else:
        obj = Get_Active_Object()
        if obj:
            obj.hide_set(False)
        else:
            print("Warning: No object selected to show.")
##########################################################################

# Parameters: ref (str or bpy.types.Object, optional) - Object to show; None for selected object.
# Usage: Alias for Show_Object, shows an object in the viewport.
def UnHide_Object(ref=None):
    """Unhide an object in the viewport (alias for Show_Object)."""
    Show_Object(ref)
##########################################################################

# Parameters: ref (str or bpy.types.Object) - Object to hide in viewport.
# Usage: Hides an object in the viewport.
def Hide_in_Viewport(ref):
    """Hide an object in the viewport."""
    obj = Get_Object_Any(ref) if is_string(ref) else ref
    obj.hide_viewport = True
##########################################################################

# Parameters: ref (str or bpy.types.Object) - Object to show in viewport.
# Usage: Shows an object in the viewport.
def Show_in_Viewport(ref):
    """Show an object in the viewport."""
    obj = Get_Object_Any(ref) if is_string(ref) else ref
    obj.hide_viewport = False
##########################################################################

# Parameters: ref (str or bpy.types.Object) - Object to hide in render.
# Usage: Hides an object in the render.
def Hide_in_Render(ref):
    """Hide an object in the render."""
    obj = Get_Object_Any(ref) if is_string(ref) else ref
    obj.hide_render = True
##########################################################################

# Parameters: ref (str or bpy.types.Object) - Object to show in render.
# Usage: Shows an object in the render.
def Show_in_Render(ref):
    """Show an object in the render."""
    obj = Get_Object_Any(ref) if is_string(ref) else ref
    obj.hide_render = False
##########################################################################

# Parameters: ref (str or bpy.types.Object) - Object to set display as bounds.
# Usage: Sets the object's display type to 'BOUNDS'.
def Display_as_Bounds(ref):
    """Set object display type to bounds."""
    obj = Get_Object_Any(ref) if is_string(ref) else ref
    obj.display_type = 'BOUNDS'
##########################################################################

# Parameters: ref (str or bpy.types.Object) - Object to set display as textured.
# Usage: Sets the object's display type to 'TEXTURED'.
def Display_as_Textured(ref):
    """Set object display type to textured."""
    obj = Get_Object_Any(ref) if is_string(ref) else ref
    obj.display_type = 'TEXTURED'
##########################################################################

# Parameters: ref (str or bpy.types.Object) - Object to set display as solid.
# Usage: Sets the object's display type to 'SOLID'.
def Display_as_Solid(ref):
    """Set object display type to solid."""
    obj = Get_Object_Any(ref) if is_string(ref) else ref
    obj.display_type = 'SOLID'
##########################################################################

# Parameters: ref (str or bpy.types.Object) - Object to set display as wire.
# Usage: Sets the object's display type to 'WIRE'.
def Display_as_Wire(ref):
    """Set object display type to wire."""
    obj = Get_Object_Any(ref) if is_string(ref) else ref
    obj.display_type = 'WIRE'
##########################################################################

# Parameters: oba (str or bpy.types.Object, optional) - Object to translate; loc (list or tuple, optional) - New location (x, y, z).
# Usage: Translates an object's location. If no location provided, returns current location; if no object, uses selected object.
def Trans_Location(oba=None, loc=None):
    """Translate an object's location or get current location."""
    obj = Get_Object_Any(oba)
    obj_provided = obj is not None
    loc_provided = loc is not None
    if obj_provided and loc_provided:
        obj.location = Vector((loc[0], loc[1], loc[2]))
    elif obj_provided:
        return obj.location
    elif loc_provided:
        obj = Get_Active_Object()
        obj.location = Vector((loc[0], loc[1], loc[2]))
    else:
        return Get_Active_Object().location
##########################################################################

# Parameters: oba (str or bpy.types.Object, optional) - Object to rotate; rot (list or tuple, optional) - New rotation in radians (x, y, z).
# Usage: Rotates an object's Euler rotation. If no rotation provided, returns current rotation; if no object, uses selected object.
def Trans_Rotation(oba=None, rot=None):
    """Rotate an object's Euler rotation or get current rotation."""
    obj = Get_Object_Any(oba)
    obj_provided = obj is not None
    rot_provided = rot is not None
    if obj_provided and rot_provided:
        obj.rotation_euler = (rot[0], rot[1], rot[2])
    elif obj_provided:
        return obj.rotation_euler
    elif rot_provided:
        obj = Get_Active_Object()
        obj.rotation_euler = (rot[0], rot[1], rot[2])
    else:
        return Get_Active_Object().rotation_euler
##########################################################################

# Parameters: oba (str or bpy.types.Object, optional) - Object to scale; scale (list or tuple, optional) - New scale (x, y, z).
# Usage: Scales an object. If no scale provided, returns current scale; if no object, uses selected object.
def Trans_Scale(oba=None, scale=None):
    """Scale an object or get current scale."""
    obj = Get_Object_Any(oba)
    obj_provided = obj is not None
    scale_provided = scale is not None
    if obj_provided and scale_provided:
        obj.scale = (scale[0], scale[1], scale[2])
    elif obj_provided:
        return obj.scale
    elif scale_provided:
        obj = Get_Active_Object()
        obj.scale = (scale[0], scale[1], scale[2])
    else:
        return Get_Active_Object().scale
##########################################################################

# Parameters: oba (str or bpy.types.Object, optional) - Object to apply location to.
# Usage: Applies location transformation to an object.
def Apply_Location(oba=None):
    """Apply location transformation to an object."""
    ref = Get_Object_Any(oba)
    if ref is not None:
        Deselect_All()
        Select_Object(ref)
        bpy.ops.object.transform_apply(location=True, rotation=False, scale=False)
##########################################################################

# Parameters: oba (str or bpy.types.Object, optional) - Object to apply rotation to.
# Usage: Applies rotation transformation to an object.
def Apply_Rotation(oba=None):
    """Apply rotation transformation to an object."""
    ref = Get_Object_Any(oba)
    if ref is not None:
        Deselect_All()
        Select_Object(ref)
        bpy.ops.object.transform_apply(location=False, rotation=True, scale=False)
##########################################################################

# Parameters: oba (str or bpy.types.Object, optional) - Object to apply scale to.
# Usage: Applies scale transformation to an object.
def Apply_Scale(oba=None):
    """Apply scale transformation to an object."""
    ref = Get_Object_Any(oba)
    if ref is not None:
        Deselect_All()
        Select_Object(ref)
        bpy.ops.object.transform_apply(location=False, rotation=False, scale=True)
##########################################################################

# Parameters: oba (str or bpy.types.Object, optional) - Object to apply all transforms to.
# Usage: Applies all transformations (location, rotation, scale) to an object.
def Apply_all_Transforms(oba=None):
    """Apply all transformations to an object."""
    ref = Get_Object_Any(oba)
    if ref is not None:
        Deselect_All()
        Select_Object(ref)
        bpy.ops.object.transform_apply(location=True, rotation=True, scale=True)
##########################################################################

# Parameters: oba (str or bpy.types.Object, optional) - Object to apply rotation and scale to.
# Usage: Applies rotation and scale transformations to an object.
def Apply_Rotation_and_Scale(oba=None):
    """Apply rotation and scale transformations to an object."""
    ref = Get_Object_Any(oba)
    if ref is not None:
        Deselect_All()
        Select_Object(ref)
        bpy.ops.object.transform_apply(location=False, rotation=True, scale=True)
##########################################################################

# Parameters: vec (list or tuple) - Translation vector (x, y, z); ref (str or bpy.types.Object, optional) - Object to translate.
# Usage: Translates an object by the given vector.
def Translate_Vector(vec, ref=None):
    """Translate an object by a vector."""
    obj = Get_Object_Any(ref) if ref else Get_Active_Object()
    obj.location[0] += vec[0]
    obj.location[1] += vec[1]
    obj.location[2] += vec[2]
##########################################################################

# Parameters: val (float) - Translation value; axis (mathutils.Vector) - Axis vector; ref (str or bpy.types.Object, optional) - Object to translate.
# Usage: Translates an object along a normalized axis by the given value.
def Translate_Along_Axis(val, axis: Vector, ref=None):
    """Translate an object along an axis."""
    obj_ref = Get_Object_Any(ref) if ref else Get_Active_Object()
    axis.normalize()
    obj_ref.location[0] += (val * axis[0])
    obj_ref.location[1] += (val * axis[1])
    obj_ref.location[2] += (val * axis[2])
##########################################################################

# Parameters: val (float) - Translation value; ref (str or bpy.types.Object, optional) - Object to translate.
# Usage: Translates an object along the global X axis.
def Translate_Along_X(val, ref=None):
    """Translate an object along the global X axis."""
    Translate_Along_Axis(val, Vector((1.0, 0.0, 0.0)), ref)
##########################################################################

# Parameters: val (float) - Translation value; ref (str or bpy.types.Object, optional) - Object to translate.
# Usage: Translates an object along the global Y axis.
def Translate_Along_Y(val, ref=None):
    """Translate an object along the global Y axis."""
    Translate_Along_Axis(val, Vector((0.0, 1.0, 0.0)), ref)
##########################################################################

# Parameters: val (float) - Translation value; ref (str or bpy.types.Object, optional) - Object to translate.
# Usage: Translates an object along the global Z axis.
def Translate_Along_Z(val, ref=None):
    """Translate an object along the global Z axis."""
    Translate_Along_Axis(val, Vector((0.0, 0.0, 1.0)), ref)
##########################################################################

# Parameters: val (float) - Translation value; ref (str or bpy.types.Object, optional) - Object to translate.
# Usage: Translates an object along its local X axis.
def Translate_along_Local_x(val, ref=None):
    """Translate an object along its local X axis."""
    obj_ref = Get_Object_Any(ref) if ref else Get_Active_Object()
    axis = Vector((1.0, 0.0, 0.0))
    axis.rotate(obj_ref.rotation_euler)
    Translate_Along_Axis(val, axis, ref)
##########################################################################

# Parameters: val (float) - Translation value; ref (str or bpy.types.Object, optional) - Object to translate.
# Usage: Translates an object along its local Y axis.
def Translate_along_Local_y(val, ref=None):
    """Translate an object along its local Y axis."""
    obj_ref = Get_Object_Any(ref) if ref else Get_Active_Object()
    axis = Vector((0.0, 1.0, 0.0))
    axis.rotate(obj_ref.rotation_euler)
    Translate_Along_Axis(val, axis, ref)
##########################################################################

# Parameters: val (float) - Translation value; ref (str or bpy.types.Object, optional) - Object to translate.
# Usage: Translates an object along its local Z axis.
def Translate_along_Local_z(val, ref=None):
    """Translate an object along its local Z axis."""
    obj_ref = Get_Object_Any(ref) if ref else Get_Active_Object()
    axis = Vector((0.0, 0.0, 1.0))
    axis.rotate(obj_ref.rotation_euler)
    Translate_Along_Axis(val, axis, ref)
##########################################################################

# Parameters: vec (list or tuple) - Rotation vector in radians (x, y, z); ref (str or bpy.types.Object, optional) - Object to rotate.
# Usage: Rotates an object by adding the given vector to its Euler rotation.
def Rotate_Vector(vec, ref=None):
    """Rotate an object by a vector."""
    obj = Get_Object_Any(ref) if ref else Get_Active_Object()
    obj.rotation_euler[0] += vec[0]
    obj.rotation_euler[1] += vec[1]
    obj.rotation_euler[2] += vec[2]
##########################################################################

# Parameters: deg (float) - Rotation degrees; axis (mathutils.Vector) - Rotation axis; obj (str or bpy.types.Object, optional) - Object to rotate; point (mathutils.Vector, optional) - Pivot point.
# Usage: Rotates an object around a normalized axis by the given degrees, with optional pivot point.
def Rotate_around_Axis(deg, axis, obj=None, point=None):
    """Rotate an object around an axis."""
    obj_ref = Get_Object_Any(obj) if obj else Get_Active_Object()
    point_ref = None
    if point is None:
        tool_settings = Get_Scene().tool_settings
        if tool_settings.transform_pivot_point == 'MEDIAN_POINT':
            point_ref = obj_ref.location
        elif tool_settings.transform_pivot_point == 'CURSOR':
            point_ref = bpy.context.scene.cursor.location
        else:
            point_ref = obj_ref.location
    else:
        point_ref = point
    axis.normalize()
    mat = Matrix.Translation(point_ref) @ Matrix.Rotation(m.radians(deg), 4, axis) @ Matrix.Translation(-point_ref)
    obj_ref.matrix_world = mat @ obj_ref.matrix_world
##########################################################################

# Parameters: deg (float) - Rotation degrees; obj (str or bpy.types.Object, optional) - Object to rotate; point (mathutils.Vector, optional) - Pivot point.
# Usage: Rotates an object around the global X axis.
def Rotate_around_Global_x(deg, obj=None, point=None):
    """Rotate an object around the global X axis."""
    Rotate_around_Axis(deg, Vector((1.0, 0.0, 0.0)), obj, point)
##########################################################################

# Parameters: deg (float) - Rotation degrees; obj (str or bpy.types.Object, optional) - Object to rotate; point (mathutils.Vector, optional) - Pivot point.
# Usage: Rotates an object around the global Y axis.
def Rotate_around_Global_y(deg, obj=None, point=None):
    """Rotate an object around the global Y axis."""
    Rotate_around_Axis(deg, Vector((0.0, 1.0, 0.0)), obj, point)
##########################################################################

# Parameters: deg (float) - Rotation degrees; obj (str or bpy.types.Object, optional) - Object to rotate; point (mathutils.Vector, optional) - Pivot point.
# Usage: Rotates an object around the global Z axis.
def Rotate_around_Global_z(deg, obj=None, point=None):
    """Rotate an object around the global Z axis."""
    Rotate_around_Axis(deg, Vector((0.0, 0.0, 1.0)), obj, point)
##########################################################################

# Parameters: deg (float) - Rotation degrees; obj (str or bpy.types.Object, optional) - Object to rotate; point (mathutils.Vector, optional) - Pivot point.
# Usage: Rotates an object around the global X axis (alias for Rotate_around_Global_x).
def Rotate_around_x(deg, obj=None, point=None):
    """Rotate an object around the global X axis (alias)."""
    Rotate_around_Global_x(deg, obj, point)
##########################################################################

# Parameters: deg (float) - Rotation degrees; obj (str or bpy.types.Object, optional) - Object to rotate; point (mathutils.Vector, optional) - Pivot point.
# Usage: Rotates an object around the global Y axis (alias for Rotate_around_Global_y).
def Rotate_around_y(deg, obj=None, point=None):
    """Rotate an object around the global Y axis (alias)."""
    Rotate_around_Global_y(deg, obj, point)
##########################################################################

# Parameters: deg (float) - Rotation degrees; obj (str or bpy.types.Object, optional) - Object to rotate; point (mathutils.Vector, optional) - Pivot point.
# Usage: Rotates an object around the global Z axis (alias for Rotate_around_Global_z).
def Rotate_around_z(deg, obj=None, point=None):
    """Rotate an object around the global Z axis (alias)."""
    Rotate_around_Global_z(deg, obj, point)
##########################################################################

# Parameters: deg (float) - Rotation degrees; obj (str or bpy.types.Object, optional) - Object to rotate; point (mathutils.Vector, optional) - Pivot point.
# Usage: Rotates an object around its local X axis.
def Rotate_around_Local_x(deg, obj=None, point=None):
    """Rotate an object around its local X axis."""
    obj_ref = Get_Object_Any(obj) if obj else Get_Active_Object()
    axis = Vector((1.0, 0.0, 0.0))
    axis.rotate(obj_ref.rotation_euler)
    Rotate_around_Axis(deg, axis, obj, point)
##########################################################################

# Parameters: deg (float) - Rotation degrees; obj (str or bpy.types.Object, optional) - Object to rotate; point (mathutils.Vector, optional) - Pivot point.
# Usage: Rotates an object around its local Y axis.
def Rotate_around_Local_y(deg, obj=None, point=None):
    """Rotate an object around its local Y axis."""
    obj_ref = Get_Object_Any(obj) if obj else Get_Active_Object()
    axis = Vector((0.0, 1.0, 0.0))
    axis.rotate(obj_ref.rotation_euler)
    Rotate_around_Axis(deg, axis, obj, point)
##########################################################################

# Parameters: deg (float) - Rotation degrees; obj (str or bpy.types.Object, optional) - Object to rotate; point (mathutils.Vector, optional) - Pivot point.
# Usage: Rotates an object around its local Z axis.
def Rotate_around_Local_z(deg, obj=None, point=None):
    """Rotate an object around its local Z axis."""
    obj_ref = Get_Object_Any(obj) if obj else Get_Active_Object()
    axis = Vector((0.0, 0.0, 1.0))
    axis.rotate(obj_ref.rotation_euler)
    Rotate_around_Axis(deg, axis, obj, point)
##########################################################################

# Parameters: rot (mathutils.Euler) - Euler rotation to reverse.
# Usage: Reverses the Euler rotation by negating axes and changing order to 'ZYX'. Returns True if successful, False if rot is None.
def Revese_Rotation_on_Euler(rot: Euler):
    """Reverse an Euler rotation."""
    if rot is not None:
        temp = rot
        temp.x *= -1
        temp.y *= -1
        temp.z *= -1
        temp.order = 'ZYX'
        return True
    return False
##########################################################################

# Parameters: vec (list or tuple) - Scale vector (x, y, z); ref (str or bpy.types.Object, optional) - Object to scale.
# Usage: Scales an object by multiplying its scale by the given vector.
def Scale_Vector(vec, ref=None):
    """Scale an object by a vector."""
    obj = Get_Object_Any(ref) if ref else Get_Active_Object()
    obj.scale[0] *= vec[0]
    obj.scale[1] *= vec[1]
    obj.scale[2] *= vec[2]
##########################################################################

# Parameters: val (float) - Uniform scale value; ref (str or bpy.types.Object, optional) - Object to scale; point (mathutils.Vector, optional) - Pivot point.
# Usage: Uniformly scales an object by the given value.
def Scale_Uniform(val, ref=None, point=None):
    """Uniformly scale an object."""
    Scale_Vector(Vector((val, val, val)), ref)
##########################################################################

# Parameters: factor (float) - Scale factor; axis (mathutils.Vector) - Axis to scale along; ref (str or bpy.types.Object, optional) - Object to scale; point (mathutils.Vector, optional) - Pivot point.
# Usage: Scales an object along a normalized axis, adjusting location to maintain pivot.
def Scale_along_Axis(factor, axis, ref=None, point=None):
    """Scale an object along an axis."""
    obj = Get_Object_Any(ref) if ref else Get_Active_Object()
    point_ref = None
    if point is None:
        tool_settings = Get_Scene().tool_settings
        if tool_settings.transform_pivot_point == 'MEDIAN_POINT':
            point_ref = obj.location
        elif tool_settings.transform_pivot_point == 'CURSOR':
            point_ref = bpy.context.scene.cursor.location
        else:
            point_ref = obj.location
    else:
        point_ref = point
    axis.normalize()
    temp = Vector()
    temp[0] = 1 + ((factor - 1) / (1 - 0)) * (axis[0] - 0)
    temp[1] = 1 + ((factor - 1) / (1 - 0)) * (axis[1] - 0)
    temp[2] = 1 + ((factor - 1) / (1 - 0)) * (axis[2] - 0)
    obj.scale[0] *= temp[0]
    obj.scale[1] *= temp[1]
    obj.scale[2] *= temp[2]
    axis.rotate(obj.rotation_euler)
    axis.normalize()
    fac = (obj.location - point_ref).dot(axis) * (factor - 1)
    Translate_Along_Axis(fac, axis, obj)
##########################################################################

# Parameters: factor (float) - Scale factor; ref (str or bpy.types.Object, optional) - Object to scale; point (mathutils.Vector, optional) - Pivot point.
# Usage: Scales an object along the X axis.
def Scale_along_x(factor, ref=None, point=None):
    """Scale an object along the X axis."""
    Scale_along_Axis(factor, Vector((1.0, 0.0, 0.0)), ref, point)
##########################################################################

# Parameters: factor (float) - Scale factor; ref (str or bpy.types.Object, optional) - Object to scale; point (mathutils.Vector, optional) - Pivot point.
# Usage: Scales an object along the Y axis.
def Scale_along_y(factor, ref=None, point=None):
    """Scale an object along the Y axis."""
    Scale_along_Axis(factor, Vector((0.0, 1.0, 0.0)), ref, point)
##########################################################################

# Parameters: factor (float) - Scale factor; ref (str or bpy.types.Object, optional) - Object to scale; point (mathutils.Vector, optional) - Pivot point.
# Usage: Scales an object along the Z axis.
def Scale_along_z(factor, ref=None, point=None):
    """Scale an object along the Z axis."""
    Scale_along_Axis(factor, Vector((0.0, 0.0, 1.0)), ref, point)
##########################################################################

# Parameters: factor (float) - Scale factor; ref (str or bpy.types.Object, optional) - Object to scale; point (mathutils.Vector, optional) - Pivot point.
# Usage: Scales an object along its local X axis.
def Scale_along_Local_x(factor, ref=None, point=None):
    """Scale an object along its local X axis."""
    Scale_along_Axis(factor, Vector((1.0, 0.0, 0.0)), ref, point)
##########################################################################

# Parameters: factor (float) - Scale factor; ref (str or bpy.types.Object, optional) - Object to scale; point (mathutils.Vector, optional) - Pivot point.
# Usage: Scales an object along its local Y axis.
def Scale_along_Local_y(factor, ref=None, point=None):
    """Scale an object along its local Y axis."""
    Scale_along_Axis(factor, Vector((0.0, 1.0, 0.0)), ref, point)
##########################################################################

# Parameters: factor (float) - Scale factor; ref (str or bpy.types.Object, optional) - Object to scale; point (mathutils.Vector, optional) - Pivot point.
# Usage: Scales an object along its local Z axis.
def Scale_along_Local_z(factor, ref=None, point=None):
    """Scale an object along its local Z axis."""
    Scale_along_Axis(factor, Vector((0.0, 0.0, 1.0)), ref, point)
##########################################################################

# Parameters: factor (float) - Scale factor; axis (mathutils.Vector) - Global axis to scale along; ref (str or bpy.types.Object, optional) - Object to scale; pointref (mathutils.Vector, optional) - Pivot point.
# Usage: Scales an object along a global axis, adjusting matrix for pivot.
def Scale_along_Global_axis(factor, axis: Vector, ref=None, pointref=None):
    """Scale an object along a global axis."""
    obj = Get_Object_Any(ref) if ref else Get_Active_Object()
    point = None
    if pointref is None:
        tool_settings = Get_Scene().tool_settings
        if tool_settings.transform_pivot_point == 'MEDIAN_POINT':
            point = obj.location
        elif tool_settings.transform_pivot_point == 'CURSOR':
            point = bpy.context.scene.cursor.location
        else:
            point = obj.location
    else:
        point = pointref
    loc, rot, scale = obj.matrix_world.decompose()
    loc_mat = Matrix.Translation(loc)
    new_loc_mat = loc_mat.copy()
    new_loc_mat.invert()
    obj.matrix_world = new_loc_mat @ obj.matrix_world
    orig_rot = obj.rotation_euler.copy()
    orig_loc = obj.location
    obj.matrix_world = Matrix.Scale(factor, 4, axis) @ obj.matrix_world
    obj.matrix_world = loc_mat @ obj.matrix_world
    obj.rotation_euler = orig_rot
    fac = (obj.location - point).dot(axis) * (factor - 1)
    Translate_Along_Axis(fac, axis, obj)
##########################################################################

# Parameters: factor (float) - Scale factor; ref (str or bpy.types.Object, optional) - Object to scale; point (mathutils.Vector, optional) - Pivot point.
# Usage: Scales an object along the global X axis.
def Scale_along_Global_x(factor, ref=None, point=None):
    """Scale an object along the global X axis."""
    Scale_along_Global_axis(factor, Vector((1.0, 0.0, 0.0)), ref, point)
##########################################################################

# Parameters: factor (float) - Scale factor; ref (str or bpy.types.Object, optional) - Object to scale; point (mathutils.Vector, optional) - Pivot point.
# Usage: Scales an object along the global Y axis.
def Scale_along_Global_y(factor, ref=None, point=None):
    """Scale an object along the global Y axis."""
    Scale_along_Global_axis(factor, Vector((0.0, 1.0, 0.0)), ref, point)
##########################################################################

# Parameters: factor (float) - Scale factor; ref (str or bpy.types.Object, optional) - Object to scale; point (mathutils.Vector, optional) - Pivot point.
# Usage: Scales an object along the global Z axis.
def Scale_along_Global_z(factor, ref=None, point=None):
    """Scale an object along the global Z axis."""
    Scale_along_Global_axis(factor, Vector((0.0, 0.0, 1.0)), ref, point)
##########################################################################

# Parameters: fac (float) - Scale factor; ref (str or bpy.types.Object, optional) - Object to scale; point (mathutils.Vector, optional) - Pivot point.
# Usage: Scales an object perpendicular to the X axis, adjusting location.
def Scale_perpendicular_to_X(fac, ref=None, point=None):
    """Scale an object perpendicular to the X axis."""
    Scale_Vector(Vector((1.0, fac, fac)), ref)
    obj = Get_Object_Any(ref) if ref else Get_Active_Object()
    point_ref = None
    if point is None:
        tool_settings = Get_Scene().tool_settings
        if tool_settings.transform_pivot_point == 'MEDIAN_POINT':
            point_ref = obj.location
        elif tool_settings.transform_pivot_point == 'CURSOR':
            point_ref = bpy.context.scene.cursor.location
        else:
            point_ref = obj.location
    else:
        point_ref = point
    axis = (obj.location - point_ref) * Vector((0.0, 1.0, 1.0))
    factor = axis.magnitude * (fac - 1)
    Translate_Along_Axis(factor, axis, obj)
##########################################################################

# Parameters: fac (float) - Scale factor; ref (str or bpy.types.Object, optional) - Object to scale; point (mathutils.Vector, optional) - Pivot point.
# Usage: Scales an object perpendicular to the Y axis, adjusting location.
def Scale_perpendicular_to_Y(fac, ref=None, point=None):
    """Scale an object perpendicular to the Y axis."""
    Scale_Vector(Vector((fac, 1.0, fac)), ref)
    obj = Get_Object_Any(ref) if ref else Get_Active_Object()
    point_ref = None
    if point is None:
        tool_settings = Get_Scene().tool_settings
        if tool_settings.transform_pivot_point == 'MEDIAN_POINT':
            point_ref = obj.location
        elif tool_settings.transform_pivot_point == 'CURSOR':
            point_ref = bpy.context.scene.cursor.location
        else:
            point_ref = obj.location
    else:
        point_ref = point
    axis = (obj.location - point_ref) * Vector((1.0, 0.0, 1.0))
    factor = axis.magnitude * (fac - 1)
    Translate_Along_Axis(factor, axis, obj)
##########################################################################

# Parameters: fac (float) - Scale factor; ref (str or bpy.types.Object, optional) - Object to scale; point (mathutils.Vector, optional) - Pivot point.
# Usage: Scales an object perpendicular to the Z axis, adjusting location.
def Scale_perpendicular_to_Z(fac, ref=None, point=None):
    """Scale an object perpendicular to the Z axis."""
    Scale_Vector(Vector((fac, fac, 1.0)), ref)
    obj = Get_Object_Any(ref) if ref else Get_Active_Object()
    point_ref = None
    if point is None:
        tool_settings = Get_Scene().tool_settings
        if tool_settings.transform_pivot_point == 'MEDIAN_POINT':
            point_ref = obj.location
        elif tool_settings.transform_pivot_point == 'CURSOR':
            point_ref = bpy.context.scene.cursor.location
        else:
            point_ref = obj.location
    else:
        point_ref = point
    axis = (obj.location - point_ref) * Vector((1.0, 1.0, 0.0))
    factor = axis.magnitude * (fac - 1)
    Translate_Along_Axis(factor, axis, obj)
##########################################################################

# Parameters: None.
# Usage: Snaps the selection to the 3D cursor without offset.
def Selection_to_Cursor_without_Offset():
    """Snap selection to cursor without offset."""
    bpy.ops.view3d.snap_selected_to_cursor(use_offset=False)
##########################################################################

# Parameters: None.
# Usage: Snaps the selection to the 3D cursor with offset.
def Selection_to_Cursor_with_Offset():
    """Snap selection to cursor with offset."""
    bpy.ops.view3d.snap_selected_to_cursor(use_offset=True)
##########################################################################

# Parameters: None.
# Usage: Snaps the 3D cursor to the world origin.
def Cursor_to_World_Origin():
    """Snap cursor to world origin."""
    bpy.ops.view3d.snap_cursor_to_center()
##########################################################################

# Parameters: None.
# Usage: Snaps the 3D cursor to the selected object.
def Cursor_to_Selection():
    """Snap cursor to selection."""
    bpy.ops.view3d.snap_cursor_to_selected()
##########################################################################

# Parameters: None.
# Usage: Snaps the 3D cursor to the active object.
def Cursor_to_Active():
    """Snap cursor to active object."""
    bpy.ops.view3d.snap_cursor_to_active()
##########################################################################

# Parameters: None.
# Usage: Snaps the selection to the grid.
def Selection_to_Grid():
    """Snap selection to grid."""
    bpy.ops.view3d.snap_selected_to_grid()
##########################################################################

# Parameters: None.
# Usage: Snaps the selection to the active object.
def Selection_to_Active():
    """Snap selection to active object."""
    bpy.ops.view3d.snap_selected_to_active()
##########################################################################

# Parameters: None.
# Usage: Snaps the 3D cursor to the grid.
def Cursor_to_Grid():
    """Snap cursor to grid."""
    bpy.ops.view3d.snap_cursor_to_grid()
##########################################################################

# Parameters: None.
# Usage: Returns the 3D cursor location as a Vector.
def Get_Cursor_Location():
    """Get the 3D cursor location."""
    return bpy.context.scene.cursor.location
##########################################################################

# Parameters: newloc (mathutils.Vector) - New cursor location.
# Usage: Sets the 3D cursor location.
def Set_Cursor_Location(newloc):
    """Set the 3D cursor location."""
    bpy.context.scene.cursor.location = newloc
##########################################################################

# Parameters: None.
# Usage: Returns the 3D cursor rotation as an Euler.
def Get_Cursor_Rotation():
    """Get the 3D cursor rotation."""
    return bpy.context.scene.cursor.rotation_euler
##########################################################################

# Parameters: None.
# Usage: Returns the 3D cursor rotation mode.
def Get_Cursor_Rotation_Mode():
    """Get the 3D cursor rotation mode."""
    return bpy.context.scene.cursor.rotation_mode
##########################################################################

# Parameters: None.
# Usage: Sets the transform pivot point to the 3D cursor.
def Pivot_Point_to_Cursor():
    """Set pivot point to cursor."""
    Get_Scene().tool_settings.transform_pivot_point = 'CURSOR'
##########################################################################

# Parameters: None.
# Usage: Sets the transform pivot point to the median point.
def Pivot_Point_to_Median():
    """Set pivot point to median."""
    Get_Scene().tool_settings.transform_pivot_point = 'MEDIAN_POINT'
##########################################################################

# Parameters: None.
# Usage: Sets the transform pivot point to individual origins.
def Pivot_Point_to_individual_Origin():
    """Set pivot point to individual origins."""
    Get_Scene().tool_settings.transform_pivot_point = 'INDIVIDUAL_ORIGINS'
##########################################################################

# Parameters: None.
# Usage: Sets the transform pivot point to the active element.
def Pivot_Point_to_Active_Element():
    """Set pivot point to active element."""
    Get_Scene().tool_settings.transform_pivot_point = 'ACTIVE_ELEMENT'
##########################################################################

# Parameters: None.
# Usage: Sets the transform pivot point to the bounding box center.
def Pivot_Point_to_bounding_Box_Center():
    """Set pivot point to bounding box center."""
    Get_Scene().tool_settings.transform_pivot_point = 'BOUNDING_BOX_CENTER'
##########################################################################

# Parameters: ref (str or bpy.types.Object, optional) - Object to set origin to center of mass.
# Usage: Sets the origin of an object to its center of mass (surface).
def Origin_to_Centermass(ref=None):
    """Set origin to center of mass (surface)."""
    obj_ref = Get_Object_Any(ref) if ref else Get_Active_Object()
    if obj_ref:
        Select_Object(obj_ref)
        bpy.ops.object.origin_set(type='ORIGIN_CENTER_OF_MASS')
    else:
        print("Warning: No object specified for origin to center of mass.")
##########################################################################

# Parameters: ref (str or bpy.types.Object, optional) - Object to set origin to center of volume.
# Usage: Sets the origin of an object to its center of volume.
def Origin_to_Center_Volume(ref=None):
    """Set origin to center of volume."""
    obj_ref = Get_Object_Any(ref) if ref else Get_Active_Object()
    if obj_ref:
        Select_Object(obj_ref)
        bpy.ops.object.origin_set(type='ORIGIN_CENTER_OF_VOLUME')
    else:
        print("Warning: No object specified for origin to center of volume.")
##########################################################################
# Parameters: step (int) - Frame step for baking (default: 1).
# Usage: Bakes rigid body simulation to keyframes for the current scene’s frame range. Requires a rigid body world and objects.
def RB_Bakeonly(step=1):
    """Bake rigid body simulation to keyframes for the current scene's frame range.
    Warning: Rigid body system is legacy in Blender 4.2+. Consider using simulation nodes for future versions.
    """
    tm = Start_Timer()
    print("Starting rigid body bake...")

    # Check for rigid body world
    if bpy.context.scene.rigidbody_world is None:
        print("Error: No rigid body world found. Please set up a rigid body simulation.")
        End_Timer(tm)
        return

    # Check for rigid body objects
    rb_objects = [obj for obj in bpy.context.scene.objects if obj.rigid_body is not None]
    if not rb_objects:
        print("Error: No rigid body objects found in the scene.")
        End_Timer(tm)
        return

    # Use scene frame range for baking
    frame_start = bpy.context.scene.frame_start
    frame_end = bpy.context.scene.frame_end

    try:
        # Free existing bakes (if any)
        bpy.ops.ptcache.free_bake_all()
        # Bake to keyframes
        bpy.ops.rigidbody.bake_to_keyframes(frame_start=frame_start, frame_end=frame_end, step=step)
        print("Rigid body bake completed successfully.")
    except Exception as e:
        print(f"Error during rigid body bake: {str(e)}")
    finally:
        End_Timer(tm)
##########################################################################

# Parameters: None.
# Usage: Adds rigid body properties to selected objects intelligently, based on object names. Skips existing rigid bodies.
def RB_RigidBodyAdd_Intelligent():
    """Add rigid body properties to selected objects intelligently, based on object names.
    Warning: Rigid body system is legacy in Blender 4.2+. Consider using simulation nodes.
    """
    tm = Start_Timer()
    print("Adding intelligent rigid body properties to selected objects...")

    # Check for rigid body world; create if missing
    if bpy.context.scene.rigidbody_world is None:
        bpy.ops.rigidbody.world_add()
        print("Created new rigid body world.")

    # Store selected objects
    saved_selection = Remember_Selected()
    selected_objects = saved_selection[1:]  # Skip active object at index 0
    if saved_selection[1:] == []:
        print("Error: No objects selected.")
        End_Timer(tm)
        return

    # Process each object
    failed_objects = []
    for obj in selected_objects:
        if Is_RigidBody(obj):
            continue  # Skip objects with existing rigid body
        try:
            RB_Add_Single_Object(obj)
        except Exception as e:
            print(f"Error adding rigid body to {obj.name}: {str(e)}")
            failed_objects.append(obj.name)

    # Restore original selection
    Restore_Selected(saved_selection)

    # Report results
    if failed_objects:
        print(f"Failed to add rigid body to: {', '.join(failed_objects)}")
    print(f"Rigid body properties added to {len(selected_objects) - len(failed_objects)} objects.")
    End_Timer(tm)
##########################################################################

# Parameters: obj (str or bpy.types.Object) - Object to add rigid body to.
# Usage: Adds rigid body properties to a single object based on its name. Skips if object already has rigid body or doesn’t exist.
def RB_Add_Single_Object(obj):
    """Add rigid body properties to a single object based on its name.
    Warning: Rigid body system is legacy in Blender 4.2+. Consider using simulation nodes.
    """
    if not Exist_Object(obj):
        print(f"Error: Object {obj} does not exist.")
        return

    if Is_RigidBody(obj):
        return  # Skip if already a rigid body

    # Store selection state
    saved_selection = Remember_Selected()
    Deselect_All()
    Set_Active_Object(obj)
    obj.select_set(True)

    # Find Outliner area for context override
    outliner_area = None
    for area in bpy.context.screen.areas:
        if area.type == 'OUTLINER':
            outliner_area = area
            break

    # Add rigid body
    typ = RB_Get_BB_by_Name(obj)
    sort = RB_Get_Sort_by_Name(obj)
    try:
        with bpy.context.temp_override(area=outliner_area) if outliner_area else contextlib.nullcontext():
            bpy.ops.rigidbody.object_add()
            obj.rigid_body.type = sort
            obj.rigid_body.restitution = 0.2
            obj.rigid_body.friction = 0.111354
            obj.rigid_body.collision_shape = typ
            RB_Set_RND_Mass(obj, 10, 500)
            RB_Deactivate(1)  # Match original default
    except Exception as e:
        print(f"Error adding rigid body to {obj.name}: {str(e)}")
    finally:
        # Restore selection
        Restore_Selected(saved_selection)
##########################################################################

# Parameters: is_active (bool) - True for active, False for passive; min_mass (float) - Minimum random mass; max_mass (float) - Maximum random mass.
# Usage: Adds rigid body properties to selected objects, setting them as active or passive with random masses.
def RB_AddSelectedObjects(is_active=True, min_mass=50, max_mass=250):
    """Add rigid body properties to selected objects.
    Warning: Rigid body system is legacy in Blender 4.2+. Consider using simulation nodes.
    """
    tm = Start_Timer()
    print("Adding rigid body properties to selected objects...")

    # Check for rigid body world; create if missing
    if bpy.context.scene.rigidbody_world is None:
        bpy.ops.rigidbody.world_add()
        print("Created new rigid body world.")

    # Store selected objects and active object
    saved_selection = Remember_Selected()
    if not saved_selection[1:]:  # Skip active object at index 0
        print("Error: No objects selected.")
        End_Timer(tm)
        return

    # Find Outliner area for context override
    outliner_area = None
    for area in bpy.context.screen.areas:
        if area.type == 'OUTLINER':
            outliner_area = area
            break

    # Process each object
    rb_type = 'ACTIVE' if is_active else 'PASSIVE'
    failed_objects = []
    for obj in saved_selection[1:]:  # Skip active object at index 0
        if obj.type not in {'MESH', 'EMPTY'}:  # Rigid bodies typically apply to meshes
            failed_objects.append(obj.name)
            continue
        if Is_RigidBody(obj):
            continue  # Skip objects already with rigid body

        # Set object as active and selected for operator
        bpy.context.view_layer.objects.active = obj
        obj.select_set(True)

        try:
            with bpy.context.temp_override(area=outliner_area) if outliner_area else contextlib.nullcontext():
                bpy.ops.rigidbody.object_add()
                obj.rigid_body.type = rb_type
                obj.rigid_body.restitution = 0.2
                obj.rigid_body.friction = 0.111354
                RB_Set_RND_Mass(obj, min_mass, max_mass)
        except Exception as e:
            print(f"Error adding rigid body to {obj.name}: {str(e)}")
            failed_objects.append(obj.name)
        finally:
            obj.select_set(False)  # Deselect after processing

    # Copy settings from first processed object to others
    if saved_selection[1:] and not Is_RigidBody(saved_selection[1]):
        first_rb_obj = next((obj for obj in saved_selection[1:] if Is_RigidBody(obj)), None)
        if first_rb_obj:
            for obj in saved_selection[1:]:
                if obj != first_rb_obj and Is_RigidBody(obj):
                    obj.select_set(True)
                    bpy.context.view_layer.objects.active = first_rb_obj
                    try:
                        with bpy.context.temp_override(area=outliner_area) if outliner_area else contextlib.nullcontext():
                            bpy.ops.rigidbody.object_settings_copy()
                    except Exception as e:
                        print(f"Error copying rigid body settings to {obj.name}: {str(e)}")
                    obj.select_set(False)

    # Restore original selection
    Restore_Selected(saved_selection)

    # Report results
    if failed_objects:
        print(f"Failed to add rigid body to: {', '.join(failed_objects)}")
    print(f"Rigid body properties added to {len(saved_selection[1:]) - len(failed_objects)} objects.")
    End_Timer(tm)
##########################################################################

# Parameters: ref (bpy.types.Object) - Object to check.
# Usage: Returns 1 if the object has rigid body properties, 0 otherwise.
def Is_RigidBody(ref):
    """Check if an object has rigid body properties."""
    return 1 if ref.rigid_body is not None else 0
##########################################################################

# Parameters: None.
# Usage: Returns 1 if the active object has rigid body properties, 0 otherwise.
def Exist_Rigid_Body():
    """Check if the active object has rigid body properties."""
    return 1 if bpy.context.object and bpy.context.object.rigid_body is not None else 0
##########################################################################

# Parameters: num (int) - Bounding box number (1-7).
# Usage: Returns a bounding box type string based on the input number.
def ChooseBB(num):
    """Choose a rigid body bounding box type by number."""
    res = Get_BB_by_Num(num)
    print(f"Chosen RB-BB is: {res}")
    return res
##########################################################################

# Parameters: val (int) - Bounding box number (default: 2).
# Usage: Returns a bounding box type string based on the input number.
def Get_BB_by_Num(val=2):
    """Get rigid body bounding box type by number."""
    BB = {
        1: 'SPHERE',
        2: 'BOX',
        3: 'CONE',
        4: 'CAPSULE',
        5: 'MESH',
        6: 'CONVEX_HULL',
        7: 'CYLINDER'
    }
    return BB.get(val, 'BOX')
##########################################################################

# Parameters: None.
# Usage: Frees all rigid body bakes in the scene.
def RB_Delete_Bakes():
    """Free all rigid body bakes."""
    if Exist_Rigid_Body():
        try:
            bpy.ops.ptcache.free_bake_all()
        except Exception as e:
            print(f"Error freeing rigid body bakes: {str(e)}")
    else:
        print("Warning: No rigid body objects found.")
##########################################################################

# Parameters: None.
# Usage: Copies rigid body settings from the active object to selected objects.
def RB_Copy_from_Active():
    """Copy rigid body settings from active object."""
    outliner_area = None
    for area in bpy.context.screen.areas:
        if area.type == 'OUTLINER':
            outliner_area = area
            break
    try:
        with bpy.context.temp_override(area=outliner_area) if outliner_area else contextlib.nullcontext():
            bpy.ops.rigidbody.object_settings_copy()
    except Exception as e:
        print(f"Error copying rigid body settings: {str(e)}")
##########################################################################

# Parameters: Steps (int) - Steps per second; Solver (int) - Solver iterations.
# Usage: Sets rigid body simulation calculation parameters.
def RB_SetCalculation(Steps, Solver):
    """Set rigid body simulation calculation parameters."""
    if bpy.context.scene.rigidbody_world:
        bpy.context.scene.rigidbody_world.steps_per_second = Steps
        bpy.context.scene.rigidbody_world.solver_iterations = Solver
    else:
        print("Error: No rigid body world found.")
##########################################################################

# Parameters: None.
# Usage: Copies rigid body settings from the active object and bakes the simulation.
def RB_Copy_from_Active_and_Bake():
    """Copy rigid body settings and bake simulation."""
    RB_Copy_from_Active()
    RB_Bakeonly()
##########################################################################

# Parameters: None.
# Usage: Bakes cloth simulations (legacy). Use with caution in Blender 4.2+.
def RB_Bake_Clothes():
    """Bake cloth simulations (legacy)."""
    try:
        bpy.ops.ptcache.free_bake_all()
        bpy.ops.ptcache.bake_all(bake=True)
    except Exception as e:
        print(f"Error baking cloth simulations: {str(e)}")
    print("Warning: Cloth baking is legacy. Consider using simulation nodes.")
##########################################################################

# Parameters: None.
# Usage: Saves the file and bakes the rigid body simulation.
def Save_and_Bake():
    """Save file and bake rigid body simulation."""
    SaveFile()
    RB_Bakeonly()
##########################################################################

# Parameters: st (int) - Start frame; en (int) - End frame.
# Usage: Sets the start and end frames for rigid body simulation baking.
def RB_Set_Start_End_Frame(st=1, en=250):
    """Set rigid body simulation frame range."""
    if bpy.context.scene.rigidbody_world:
        bpy.context.scene.rigidbody_world.point_cache.frame_start = st
        bpy.context.scene.rigidbody_world.point_cache.frame_end = en
    else:
        print("Error: No rigid body world found.")
##########################################################################

# Parameters: st (int) - Start frame; en (int) - End frame; step (int) - Frame step.
# Usage: Sets the start, end, and step for soft body simulation baking.
def SB_Set_Start_End_Frame(st=1, en=250, step=1):
    """Set soft body simulation frame range."""
    obj = bpy.context.object
    if obj and "Softbody" in obj.modifiers:
        mod = obj.modifiers["Softbody"]
        mod.point_cache.frame_start = st
        mod.point_cache.frame_end = en
        mod.point_cache.frame_step = step
    else:
        print("Error: No soft body modifier found on active object.")
##########################################################################

# Parameters: val (int, optional) - Start frame to set; None to get current start frame.
# Usage: Gets or sets the scene’s animation start frame.
def Ani_Start_Frame(val=None):
    """Get or set animation start frame."""
    if val is None:
        return Get_Scene().frame_start
    Get_Scene().frame_start = val
##########################################################################

# Parameters: val (int, optional) - End frame to set; None to get current end frame.
# Usage: Gets or sets the scene’s animation end frame.
def Ani_End_Frame(val=None):
    """Get or set animation end frame."""
    if val is None:
        return Get_Scene().frame_end
    Get_Scene().frame_end = val
##########################################################################

# Parameters: start (int, optional) - Start frame; end (int, optional) - End frame.
# Usage: Sets or gets the animation start and end frames as a list.
def Ani_Set_Frames(start=None, end=None):
    """Set or get animation start and end frames."""
    if start is None:
        return [Ani_Start_Frame(), Ani_End_Frame()]
    Ani_Start_Frame(start)
    Ani_End_Frame(end)
    return []
##########################################################################

# Parameters: val (int) - Frame step value.
# Usage: Sets the animation frame step.
def Ani_Set_Step(val):
    """Set animation frame step."""
    Get_Scene().frame_step = val
##########################################################################

# Parameters: st (int) - Start frame; en (int) - End frame.
# Usage: Sets the scene’s animation start and end frames.
def Ani_Set_Start_End_Frame(st=1, en=250):
    """Set animation start and end frames."""
    bpy.context.scene.frame_start = st
    bpy.context.scene.frame_end = en
##########################################################################

# Parameters: st (int) - Start frame; en (int) - End frame.
# Usage: Sets both animation and rigid body simulation frame ranges.
def Set_Start_End_Frames(st=1, en=250):
    """Set animation and rigid body frame ranges."""
    Ani_Set_Start_End_Frame(st, en)
    RB_Set_Start_End_Frame(st, en)
##########################################################################

# Parameters: None.
# Usage: Synchronizes rigid body simulation frames with scene frames.
def RB_Synchron_Frames():
    """Synchronize rigid body simulation frames with scene."""
    en = bpy.context.scene.frame_end
    st = bpy.context.scene.frame_start
    RB_Set_Start_End_Frame(st, en)
##########################################################################

# Parameters: None.
# Usage: Synchronizes soft body simulation frames with scene frames.
def SB_Synchron_Frames():
    """Synchronize soft body simulation frames with scene."""
    en = bpy.context.scene.frame_end
    st = bpy.context.scene.frame_start
    SB_Set_Start_End_Frame(st, en)
##########################################################################

# Parameters: Steps (int) - Steps per second; Solver (int) - Solver iterations.
# Usage: Bakes rigid body simulation with specified calculation parameters.
def RB_Bake(Steps, Solver):
    """Bake rigid body simulation with calculation parameters."""
    tm = Start_Timer()
    print(f"Started with RB-BAKE: {Steps}/{Solver}")
    try:
        bpy.ops.ptcache.free_bake_all()
        RB_SetCalculation(Steps, Solver)
        bpy.ops.rigidbody.bake_to_keyframes(frame_start=bpy.context.scene.frame_start, frame_end=bpy.context.scene.frame_end)
        print("RB-BAKE Done")
    except Exception as e:
        print(f"Error during rigid body bake: {str(e)}")
    End_Timer(tm)
##########################################################################

# Parameters: type (str) - Rigid body type ('ACTIVE' or 'PASSIVE'); bbox (str) - Collision shape; ac (int, optional) - Legacy: 0=active, 1=deactivated (deprecated); deactivate (bool) - True to deactivate.
# Usage: Adds a rigid body to the active object with specified type and collision shape.
def AddRigidBody(type, bbox, ac=None, deactivate=True):
    """Add rigid body to active object.
    Warning: Rigid body system is legacy in Blender 4.2+. Consider using simulation nodes.
    """
    if ac is not None:
        warnings.warn(
            "The 'ac' parameter is deprecated. Use 'deactivate' instead.",
            DeprecationWarning,
            stacklevel=2
        )
        deactivate = (ac == 1)
    try:
        bpy.ops.rigidbody.object_add()
        bpy.context.object.rigid_body.type = type
        bpy.context.object.rigid_body.restitution = 0.2
        bpy.context.object.rigid_body.friction = 0.111354
        bpy.context.object.rigid_body.collision_shape = bbox
        if deactivate:
            bpy.context.object.rigid_body.use_deactivation = True
            bpy.context.object.rigid_body.use_start_deactivated = True
    except Exception as e:
        print(f"Error adding rigid body: {str(e)}")
##########################################################################

# Parameters: None.
# Usage: Removes rigid body properties from selected objects.
def RB_Remove():
    """Remove rigid body properties from selected objects."""
    try:
        bpy.ops.rigidbody.objects_remove()
    except Exception as e:
        print(f"Error removing rigid body properties: {str(e)}")
##########################################################################

# Parameters: None.
# Usage: Resets the rigid body world in the scene.
def RB_Reset():
    """Reset the rigid body world."""
    try:
        bpy.ops.rigidbody.world_remove()
        bpy.ops.rigidbody.world_add()
    except Exception as e:
        print(f"Error resetting rigid body world: {str(e)}")
##########################################################################

# Parameters: None.
# Usage: Bakes rigid body transformations to keyframes for selected objects.
def RB_Bake_to_Frames():
    """Bake rigid body transformations to keyframes."""
    try:
        bpy.ops.rigidbody.bake_to_keyframes()
    except Exception as e:
        print(f"Error baking rigid body to keyframes: {str(e)}")
##########################################################################

# Parameters: None.
# Usage: Removes rigid body constraints from selected objects.
def RB_Remove_Constraint():
    """Remove rigid body constraints."""
    try:
        bpy.ops.rigidbody.constraint_remove()
    except Exception as e:
        print(f"Error removing rigid body constraints: {str(e)}")
##########################################################################

# Parameters: None.
# Usage: Returns the scene’s first frame.
def GetFirstFrame():
    """Get the scene’s first frame."""
    return bpy.context.scene.frame_start
##########################################################################

# Parameters: None.
# Usage: Returns the scene’s last frame.
def GetLastFrame():
    """Get the scene’s last frame."""
    return bpy.context.scene.frame_end
##########################################################################

# Parameters: None.
# Usage: Sets selected objects as dynamic rigid bodies.
def RB_Add_Dynamic():
    """Set selected objects as dynamic rigid bodies."""
    RB_Set_Dynamic(True)
##########################################################################

# Parameters: None.
# Usage: Removes dynamic rigid body properties from selected objects.
def RB_Remove_Dynamic():
    """Remove dynamic rigid body properties."""
    RB_Set_Dynamic(False)
##########################################################################

# Parameters: None.
# Usage: Sets selected objects as kinematic rigid bodies.
def RB_Add_Kinematic():
    """Set selected objects as kinematic rigid bodies."""
    RB_Set_Kinematic(True)
##########################################################################

# Parameters: None.
# Usage: Removes kinematic rigid body properties from selected objects.
def RB_Remove_Kinematic():
    """Remove kinematic rigid body properties."""
    RB_Set_Kinematic(False)
##########################################################################

# Parameters: t (bool) - True to enable dynamic, False to disable.
# Usage: Sets dynamic state for selected rigid body objects.
def RB_Set_Dynamic(t=True):
    """Set dynamic state for selected rigid body objects."""
    for obj in bpy.context.selected_objects:
        if Is_RigidBody(obj):
            obj.rigid_body.enabled = t
        else:
            print(f"Warning: Object {obj.name} is not a rigid body.")
##########################################################################

# Parameters: t (bool) - True to enable kinematic, False to disable; kf (int) - Keyframe number.
# Usage: Sets kinematic state for selected rigid body objects at the specified keyframe.
def RB_Set_Kinematic_at_KF(t=True, kf=1):
    """Set kinematic state for selected rigid body objects at keyframe."""
    for obj in bpy.context.selected_objects:
        RB_Set_Kinematic_obj_at_KF(obj, t, kf)
##########################################################################

# Parameters: oba (str or bpy.types.Object) - Object to set; t (bool) - True to enable kinematic; kf (int) - Keyframe number.
# Usage: Sets kinematic state for a single object at the specified keyframe.
def RB_Set_Kinematic_obj_at_KF(oba, t=True, kf=1):
    """Set kinematic state for an object at keyframe."""
    obj = Get_Object_Any(oba)
    if Is_RigidBody(obj):
        obj.rigid_body.kinematic = t
        obj.keyframe_insert(data_path="rigid_body.kinematic", frame=kf)
    else:
        print(f"Error: Object {obj.name} is not a rigid body.")
##########################################################################

# Parameters: t (bool) - True to enable dynamic; kf (int) - Keyframe number.
# Usage: Sets dynamic state for selected rigid body objects at the specified keyframe.
def RB_Set_Dynamic_at_KF(t=True, kf=1):
    """Set dynamic state for selected rigid body objects at keyframe."""
    for obj in bpy.context.selected_objects:
        if Is_RigidBody(obj):
            RB_Set_Dynamic_obj_at_KF(obj, t, kf)
        else:
            print(f"Warning: Object {obj.name} is not a rigid body.")
##########################################################################

# Parameters: oba (str or bpy.types.Object) - Object to set; t (bool) - True to enable dynamic; kf (int) - Keyframe number.
# Usage: Sets dynamic state for a single object at the specified keyframe.
def RB_Set_Dynamic_obj_at_KF(oba, t=True, kf=1):
    """Set dynamic state for an object at keyframe."""
    obj = Get_Object_Any(oba)
    if Is_RigidBody(obj):
        obj.rigid_body.enabled = t
        obj.keyframe_insert(data_path="rigid_body.enabled", frame=kf)
    else:
        print(f"Error: Object {obj.name} is not a rigid body.")
##########################################################################

# Parameters: t (bool) - True to enable dynamic; kf (int) - Keyframe number.
# Usage: Sets dynamic state for selected objects at the current keyframe.
def RB_Set_Dynamic_at_actual_KF(t=True):
    """Set dynamic state for selected objects at current keyframe."""
    kf = Get_Actual_KF()
    RB_Set_Dynamic_at_KF(t, kf)
##########################################################################

# Parameters: CA (list) - List of objects; t (bool) - True to enable dynamic; kf (int) - Keyframe number.
# Usage: Sets dynamic state for a list of objects at the specified keyframe.
def RB_Set_Dynamic_for_A_at_KF(CA, t=True, kf=1):
    """Set dynamic state for a list of objects at keyframe."""
    for obj in CA:
        if Is_RigidBody(obj):
            RB_Set_Dynamic_obj_at_KF(obj, t, kf)
        else:
            print(f"Warning: Object {obj.name} is not a rigid body.")
##########################################################################

# Parameters: t (bool) - True to enable kinematic; kf (int) - Keyframe number.
# Usage: Sets kinematic state for selected objects at the current keyframe.
def RB_Set_Kinematic_at_actual_KF(t=True):
    """Set kinematic state for selected objects at current keyframe."""
    kf = Get_Actual_KF()
    RB_Set_Kinematic_at_KF(t, kf)
##########################################################################

# Parameters: Pa (int) - Start frame; Pb (int) - End frame.
# Usage: Makes selected objects invisible and deactivates rigid body from frame Pa to Pb.
def BulletKFA(Pa, Pb):
    """Make selected objects invisible and deactivate rigid body from frame Pa to Pb."""
    RB_RigidBodyAdd_Intelligent()
    Ani_View_Selected_Objects_at_KF(False, Pa, 1)
    RB_Set_Dynamic_at_KF(False, Pb)
    RB_Set_Kinematic_at_KF(True, Pb)
    RB_Set_Dynamic_at_KF(True, Pa)
    RB_Set_Kinematic_at_KF(False, Pa)
    Ani_View_Selected_Objects_at_KF(True, Pb, 1)
    cprint("Ready")
##########################################################################

# Parameters: Pa (int) - Start frame; Pb (int) - End frame.
# Usage: Makes selected objects visible and active from frame Pa to Pb, with a two-pass execution for reliability.
def BulletKFA2(Pa, Pb):
    """Make selected objects invisible and deactivate rigid body from frame Pa to Pb (two-pass)."""
    GX = Remember_Selected()
    BulletKFA(Pa, Pb)
    Deselect_All()
    Restore_Selected(GX)
    BulletKFA(Pa, Pb)
    cprint("Ready")
##########################################################################

# Parameters: Pa (int) - Start frame; Pb (int) - End frame.
# Usage: Makes selected objects visible and active from frame Pa to Pb.
def BulletKFB(Pa, Pb):
    """Make selected objects visible and active from frame Pa to Pb."""
    Pc = Pa + 1
    RB_RigidBodyAdd_Intelligent()
    Ani_View_Selected_Objects_at_KF(False, Pa, 1)
    Ani_View_Selected_Objects_at_KF(True, Pb, 1)
    RB_Set_Dynamic_at_KF(True, Pc)
    RB_Set_Dynamic_at_KF(False, Pb)
    RB_Set_Kinematic_at_KF(False, Pc)
    RB_Set_Kinematic_at_KF(True, Pb)
    cprint("Ready")
##########################################################################

# Parameters: Pa (int) - Start frame; Pb (int) - End frame.
# Usage: Makes selected objects visible and active from frame Pa to Pb, with a two-pass execution.
def BulletKFB2(Pa, Pb):
    """Make selected objects visible and active from frame Pa to Pb (two-pass)."""
    GX = Remember_Selected()
    BulletKFB(Pa, Pb)
    Deselect_All()
    Restore_Selected(GX)
    BulletKFB(Pa, Pb)
    cprint("Ready")
##########################################################################

# Parameters: Pa (int) - Start frame; Pb (int) - End frame.
# Usage: Makes selected objects visible but inactive from frame Pa to Pb.
def BulletKFC(Pa, Pb):
    """Make selected objects visible but inactive from frame Pa to Pb."""
    RB_RigidBodyAdd_Intelligent()
    Ani_View_Selected_Objects_at_KF(False, Pa)
    RB_Set_Dynamic_at_KF(False, Pb)
    RB_Set_Dynamic_at_KF(False, Pa)
    Ani_View_Selected_Objects_at_KF(True, Pb)
    cprint("Ready")
##########################################################################

# Parameters: Pa (int) - Start frame; Pb (int) - End frame.
# Usage: Makes selected objects visible but inactive from frame Pa to Pb, with a two-pass execution.
def BulletKFC2(Pa, Pb):
    """Make selected objects visible but inactive from frame Pa to Pb (two-pass)."""
    GX = Remember_Selected()
    BulletKFC(Pa, Pb)
    Deselect_All()
    Restore_Selected(GX)
    BulletKFC(Pa, Pb)
    cprint("Ready")
##########################################################################

# Parameters: Pa (int) - Start frame; Pb (int) - End frame.
# Usage: Makes selected objects visible in render from frame Pa to Pb, keeping viewport visibility.
def BulletKFD(Pa, Pb):
    """Make selected objects visible in render from frame Pa to Pb."""
    Pc = Pa + 1
    RB_RigidBodyAdd_Intelligent()
    Ani_View_Selected_Objects_at_KF(False, Pa)
    Ani_View_Selected_Objects_at_KF(True, Pb)
    RB_Set_Dynamic_at_KF(True, Pc)
    RB_Set_Dynamic_at_KF(False, Pb)
    cprint("Ready")
##########################################################################

# Parameters: obj (str or bpy.types.Object) - Object to set; fr (int) - Frame number; ta (int) - 0 for start, 1 for target.
# Usage: Sets kinematic and dynamic states for an object at a specific frame.
def RB_Set_KinDyn_Object(obj, fr, ta=0):
    """Set kinematic and dynamic states for an object at a frame."""
    obj_ref = Get_Object_Any(obj)
    Ki = True if ta == 0 else False
    Dy = True
    RB_Set_Kinematic_obj_at_KF(obj_ref, Ki, fr)
    if Is_RigidBody(obj_ref):
        RB_Set_Dynamic_obj_at_KF(obj_ref, Dy, fr)
    else:
        print(f"Error: Object {obj_ref.name} is not a rigid body.")
##########################################################################

# Parameters: fr (int) - Frame number; ta (int) - 0 for start, 1 for target.
# Usage: Sets kinematic and dynamic states for selected objects at a specific frame.
def RB_Set_KinDyn(fr, ta=0):
    """Set kinematic and dynamic states for selected objects at a frame."""
    Ki = True if ta == 0 else False
    Dy = True
    RB_Set_Kinematic_at_KF(Ki, fr)
    RB_Set_Dynamic_at_KF(Dy, fr)
##########################################################################

# Parameters: pa (int, optional) - Legacy: 1=deactivate, 0=active (deprecated); deactivate (bool) - True to deactivate.
# Usage: Sets deactivation state for the active object’s rigid body.
def RB_Deactivate(pa=None, deactivate=True):
    """Set rigid body deactivation state."""
    if pa is not None:
        warnings.warn(
            "The 'pa' parameter is deprecated. Use 'deactivate' instead.",
            DeprecationWarning,
            stacklevel=2
        )
        deactivate = (pa == 1)
    if bpy.context.object and Is_RigidBody(bpy.context.object):
        bpy.context.object.rigid_body.use_deactivation = deactivate
        bpy.context.object.rigid_body.use_start_deactivated = deactivate
    else:
        print("Error: No valid rigid body object active.")
##########################################################################

# Parameters: typ (str) - Bounding box type code ('B', 'S', etc.).
# Usage: Converts a bounding box type code to a full type string.
def RB_Get_BB_Type(typ='B'):
    """Get rigid body bounding box type from code."""
    if typ == 'B':
        return 'BOX'
    elif typ == 'S':
        return 'SPHERE'
    elif typ == 'C':
        return 'CONVEX_HULL'
    elif typ == 'K':
        return 'CAPSULE'
    elif typ == 'O':
        return 'CONE'
    elif typ == 'Y':
        return 'CYLINDER'
    elif typ == 'M':
        return 'MESH'
    return 'MESH'
##########################################################################

# Parameters: obj (bpy.types.Object) - Object to check.
# Usage: Determines the bounding box type based on the object’s name or type.
def RB_Get_BB_by_Name(obj):
    """Determine bounding box type based on object name or type."""
    name = obj.name.lower()
    if obj.type != 'MESH':
        return 'MESH'
    if "plane" in name:
        return 'BOX'
    elif "cube" in name:
        return 'BOX'
    elif "sphere" in name or "icosphere" in name:
        return 'SPHERE'
    elif "cylinder" in name:
        return 'CYLINDER'
    elif "cone" in name:
        return 'CONE'
    elif "torus" in name or "gd_mesh" in name:
        return 'CONVEX_HULL'
    return 'MESH'
##########################################################################

# Parameters: obj (bpy.types.Object) - Object to check.
# Usage: Determines the rigid body type ('ACTIVE' or 'PASSIVE') based on the object’s name.
def RB_Get_Sort_by_Name(obj):
    """Determine rigid body type based on object name."""
    name = obj.name.lower()
    if "plane" in name:
        return 'PASSIVE'
    return 'ACTIVE'
##########################################################################

# Parameters: obj (str or bpy.types.Object) - Object to add; pa (int) - 1=active, 0=passive; val (str) - 'ACTIVE' or 'PASSIVE'.
# Usage: Adds a rigid body to a single object with specified type.
def RB_Add_Object(obj, pa=1, val='ACTIVE'):
    """Add rigid body to a single object.
    Warning: Rigid body system is legacy in Blender 4.2+. Consider using simulation nodes.
    """
    inter = Remember_Selected()
    Set_Active_Object(obj)
    obj.select_set(True)
    a = Is_RigidBody(obj)
    if a == 0:
        try:
            bpy.ops.rigidbody.object_add(type=val)
            RB_Set_RND_Mass(obj, 10, 500)
        except Exception as e:
            print(f"Error adding rigid body to {obj.name}: {str(e)}")
    Restore_Selected(inter)
##########################################################################

# Parameters: ma (float) - Mass value.
# Usage: Sets the mass for selected rigid body objects.
def RB_Set_Mass_Sel(ma=100):
    """Set mass for selected rigid body objects."""
    for obj in bpy.context.selected_objects:
        RB_Set_Mass(obj, ma)
##########################################################################

# Parameters: obj (bpy.types.Object) - Object to set; ma (float) - Mass value.
# Usage: Sets the mass for a single rigid body object.
def RB_Set_Mass(obj, ma):
    """Set mass for a rigid body object."""
    if Is_RigidBody(obj):
        cprint(obj)
        obj.rigid_body.mass = ma
    else:
        print(f"Error: Object {obj.name} is not a rigid body.")
##########################################################################

# Parameters: min (float) - Minimum mass; max (float) - Maximum mass.
# Usage: Sets random masses for selected rigid body objects.
def RB_Set_RND_Mass_Sel(min=100, max=1000):
    """Set random masses for selected rigid body objects."""
    for obj in bpy.context.selected_objects:
        RB_Set_RND_Mass(obj, min, max)
##########################################################################

# Parameters: obj (bpy.types.Object) - Object to set; min (float) - Minimum mass; max (float) - Maximum mass.
# Usage: Sets a random mass for a single rigid body object.
def RB_Set_RND_Mass(obj, min=100, max=1000):
    """Set a random mass for a rigid body object."""
    ma = r.uniform(min, max)
    RB_Set_Weight(obj, ma)
##########################################################################

# Parameters: obj (bpy.types.Object) - Object to set; weig (float) - Weight/mass value.
# Usage: Sets the weight (mass) for a rigid body object.
def RB_Set_Weight(obj, weig=1000):
    """Set weight (mass) for a rigid body object."""
    if Is_RigidBody(obj):
        obj.rigid_body.mass = weig
    else:
        print(f"Error: Object {obj.name} is not a rigid body.")
##########################################################################

# Parameters: obj (bpy.types.Object) - Object to query.
# Usage: Returns the weight (mass) of a rigid body object.
def RB_Get_Weight(obj):
    """Get weight (mass) of a rigid body object."""
    if Is_RigidBody(obj):
        return obj.rigid_body.mass
    print(f"Error: Object {obj.name} is not a rigid body.")
    return 0
##########################################################################

# Parameters: typ (str) - Bounding box type.
# Usage: Sets the bounding box type for the active rigid body object.
def RB_Set_Full_BB(typ):
    """Set rigid body bounding box type."""
    if bpy.context.object and Is_RigidBody(bpy.context.object):
        try:
            bpy.ops.rigidbody.shape_change(type=typ)
            print(f"BB was set to {typ}")
        except Exception as e:
            print(f"Error setting bounding box type: {str(e)}")
    else:
        print("Error: No valid rigid body object active.")
##########################################################################

# Parameters: typ (str) - Bounding box type code ('B', 'S', etc.).
# Usage: Sets the bounding box type for the active rigid body object using a type code.
def RB_SetBB(typ='B'):
    """Set rigid body bounding box type from code."""
    ty = RB_Get_BB_Type(typ)
    RB_Set_Full_BB(ty)
##########################################################################

# Parameters: None.
# Usage: Adds collision modifiers to selected objects.
def SB_Add_Collision_selected():
    """Add collision modifiers to selected objects."""
    inter = Remember_Selected()
    alo = list()
    for obj in bpy.context.selected_objects:
        alo.append(obj)
    Deselect_All()
    for i in alo:
        obj = i
        Set_Active_Object(obj)
        obj.select_set(True)
        Add_Active_as_Collision()
    Restore_Selected(inter)
##########################################################################

# Parameters: None.
# Usage: Removes soft body modifiers from selected objects.
def SB_Remove_SB_selected():
    """Remove soft body modifiers from selected objects."""
    inter = Remember_Selected()
    alo = list()
    for obj in bpy.context.selected_objects:
        alo.append(obj)
    Deselect_All()
    for i in alo:
        obj = i
        Set_Active_Object(obj)
        obj.select_set(True)
        Remove_Softbody_from_Active()
    Restore_Selected(inter)
##########################################################################

# Parameters: None.
# Usage: Removes collision modifiers from selected objects.
def SB_Remove_Collision_selected():
    """Remove collision modifiers from selected objects."""
    inter = Remember_Selected()
    alo = list()
    for obj in bpy.context.selected_objects:
        alo.append(obj)
    Deselect_All()
    for i in alo:
        obj = i
        Set_Active_Object(obj)
        obj.select_set(True)
        Remove_Collision_from_Active()
    Restore_Selected(inter)
##########################################################################

# Parameters: None.
# Usage: Adds a collision modifier to the active object.
def Add_Active_as_Collision():
    """Add collision modifier to active object."""
    try:
        bpy.ops.object.modifier_add(type='COLLISION')
    except Exception as e:
        print(f"Error adding collision modifier: {str(e)}")
##########################################################################

# Parameters: None.
# Usage: Removes the collision modifier from the active object.
def Remove_Collision_from_Active():
    """Remove collision modifier from active object."""
    if bpy.context.object and "Collision" in bpy.context.object.modifiers:
        bpy.ops.object.modifier_remove(modifier="Collision")
    else:
        print("Error: No collision modifier found on active object.")
##########################################################################

# Parameters: None.
# Usage: Removes the soft body modifier from the active object.
def Remove_Softbody_from_Active():
    """Remove soft body modifier from active object."""
    if bpy.context.object and "Softbody" in bpy.context.object.modifiers:
        bpy.ops.object.modifier_remove(modifier="Softbody")
    else:
        print("Error: No soft body modifier found on active object.")
##########################################################################

# Parameters: None.
# Usage: Adds soft body and collision modifiers to the active object with specific settings.
def SB_Add_Active():
    """Add soft body and collision modifiers to active object."""
    if bpy.context.object:
        try:
            bpy.ops.object.modifier_add(type='SOFT_BODY')
            bpy.ops.object.modifier_add(type='COLLISION')
            mod = bpy.context.object.modifiers["Softbody"]
            mod.settings.use_goal = False
            mod.settings.use_edges = True
            mod.settings.use_self_collision = True
            mod.settings.use_edge_collision = True
            mod.settings.use_face_collision = True
            mod.settings.use_stiff_quads = True
            mod.settings.use_auto_step = True
            mod.settings.bend = 1
            mod.settings.pull = 0.1
            mod.settings.push = 0.1
        except Exception as e:
            print(f"Error adding soft body modifier: {str(e)}")
    else:
        print("Error: No active object.")
##########################################################################

# Parameters: obj (str or bpy.types.Object) - Object to add soft body to.
# Usage: Adds soft body and collision modifiers to a specified object.
def SB_Add_Object(obj):
    """Add soft body and collision modifiers to an object."""
    inter = Remember_Selected()
    Set_Active_Object(obj)
    obj.select_set(True)
    SB_Add_Active()
    Restore_Selected(inter)
##########################################################################

# Parameters: None.
# Usage: Adds soft body and collision modifiers to all selected objects.
def SB_Add_selected():
    """Add soft body and collision modifiers to selected objects."""
    inter = Remember_Selected()
    alo = list()
    for obj in bpy.context.selected_objects:
        alo.append(obj)
    Deselect_All()
    for i in alo:
        obj = i
        Set_Active_Object(obj)
        obj.select_set(True)
        SB_Add_Active()
    Restore_Selected(inter)
##########################################################################

# Parameters: None.
# Usage: Copies soft body modifiers from the active object to selected objects.
def Copy_SB_Modifiers():
    """Copy soft body modifiers from active to selected objects."""
    try:
        bpy.ops.object.make_links_data(type='MODIFIERS')
    except Exception as e:
        print(f"Error copying soft body modifiers: {str(e)}")
##########################################################################

# Parameters: t (bool) - True to enable kinematic, False to disable.
# Usage: Sets kinematic state for selected rigid body objects.
def RB_Set_Kinematic(t=True):
    """Set kinematic state for selected rigid body objects."""
    for obj in bpy.context.selected_objects:
        if Is_RigidBody(obj):
            obj.rigid_body.kinematic = t
        else:
            print(f"Warning: Object {obj.name} is not a rigid body.")
##########################################################################
# Parameters: None.
# Usage: Sets the active object as a fluid domain with specified settings. Requires a mesh object.
def MF_SetDomain():
    """Set active object as fluid domain.
    Warning: Mantaflow is legacy in Blender 4.1+. Consider using simulation nodes for fluid simulations.
    """
    obj = bpy.context.object
    if not obj or obj.type != 'MESH':
        print("Error: No valid mesh object selected for fluid domain.")
        return
    try:
        bpy.ops.object.modifier_add(type='FLUID')
        mod = obj.modifiers["Fluid"]
        mod.fluid_type = 'DOMAIN'
        mod.domain_settings.resolution_max = 64
        mod.domain_settings.use_adaptive_timesteps = True
        mod.domain_settings.timestep = 0.01
        print(f"Set {obj.name} as fluid domain.")
    except Exception as e:
        print(f"Error setting fluid domain: {str(e)}")
##########################################################################

# Parameters: None.
# Usage: Sets the active object as a fluid inflow. Requires a mesh object.
def MF_SetInFlow():
    """Set active object as fluid inflow.
    Warning: Mantaflow is legacy in Blender 4.1+. Consider using simulation nodes.
    """
    obj = bpy.context.object
    if not obj or obj.type != 'MESH':
        print("Error: No valid mesh object selected for fluid inflow.")
        return
    try:
        bpy.ops.object.modifier_add(type='FLUID')
        mod = obj.modifiers["Fluid"]
        mod.fluid_type = 'INFLOW'
        mod.flow_settings.flow_type = 'LIQUID'
        mod.flow_settings.flow_behavior = 'INFLOW'
        print(f"Set {obj.name} as fluid inflow.")
    except Exception as e:
        print(f"Error setting fluid inflow: {str(e)}")
##########################################################################

# Parameters: None.
# Usage: Sets the active object as a fluid outflow. Requires a mesh object.
def MF_SetOutFlow():
    """Set active object as fluid outflow.
    Warning: Mantaflow is legacy in Blender 4.1+. Consider using simulation nodes.
    """
    obj = bpy.context.object
    if not obj or obj.type != 'MESH':
        print("Error: No valid mesh object selected for fluid outflow.")
        return
    try:
        bpy.ops.object.modifier_add(type='FLUID')
        mod = obj.modifiers["Fluid"]
        mod.fluid_type = 'OUTFLOW'
        print(f"Set {obj.name} as fluid outflow.")
    except Exception as e:
        print(f"Error setting fluid outflow: {str(e)}")
##########################################################################

# Parameters: None.
# Usage: Sets the active object as a fluid effector (collision). Requires a mesh object.
def MF_SetEffector():
    """Set active object as fluid effector.
    Warning: Mantaflow is legacy in Blender 4.1+. Consider using simulation nodes.
    """
    obj = bpy.context.object
    if not obj or obj.type != 'MESH':
        print("Error: No valid mesh object selected for fluid effector.")
        return
    try:
        bpy.ops.object.modifier_add(type='FLUID')
        mod = obj.modifiers["Fluid"]
        mod.fluid_type = 'EFFECTOR'
        mod.effector_settings.surface_distance = 0.001
        print(f"Set {obj.name} as fluid effector.")
    except Exception as e:
        print(f"Error setting fluid effector: {str(e)}")
##########################################################################

# Parameters: None.
# Usage: Bakes the fluid simulation for the current scene.
def MF_Bake():
    """Bake fluid simulation.
    Warning: Mantaflow is legacy in Blender 4.1+. Consider using simulation nodes.
    """
    domain = None
    for obj in bpy.context.scene.objects:
        for mod in obj.modifiers:
            if mod.type == 'FLUID' and mod.fluid_type == 'DOMAIN':
                domain = obj
                break
        if domain:
            break
    if not domain:
        print("Error: No fluid domain found in scene.")
        return
    try:
        bpy.context.view_layer.objects.active = domain
        bpy.ops.fluid.bake_data()
        print("Fluid simulation baked successfully.")
    except Exception as e:
        print(f"Error baking fluid simulation: {str(e)}")
##########################################################################

# Parameters: None.
# Usage: Frees the fluid simulation cache.
def MF_Free():
    """Free fluid simulation cache."""
    domain = None
    for obj in bpy.context.scene.objects:
        for mod in obj.modifiers:
            if mod.type == 'FLUID' and mod.fluid_type == 'DOMAIN':
                domain = obj
                break
        if domain:
            break
    if not domain:
        print("Error: No fluid domain found in scene.")
        return
    try:
        bpy.context.view_layer.objects.active = domain
        bpy.ops.fluid.free_data()
        print("Fluid simulation cache freed.")
    except Exception as e:
        print(f"Error freeing fluid cache: {str(e)}")
##########################################################################

# Parameters: None.
# Usage: Adds a particle system to the active object with default settings.
def Add_Particles_to_Active():
    """Add particle system to active object.
    Warning: Legacy particle system is deprecated in Blender 4.2+. Use geometry nodes for particles/hair.
    """
    obj = bpy.context.object
    if not obj:
        print("Error: No active object.")
        return
    try:
        bpy.ops.object.particle_system_add()
        ps = obj.particle_systems.active
        ps.settings.type = 'EMITTER'
        ps.settings.emit_from = 'FACE'
        ps.settings.count = 100
        ps.settings.frame_start = 1
        ps.settings.frame_end = 250
        ps.settings.lifetime = 250
        ps.settings.normal_factor = 0.1
        if bpy.data.materials:
            ps.settings.material = 1
        print(f"Added particle system to {obj.name}.")
    except Exception as e:
        print(f"Error adding particle system: {str(e)}")
##########################################################################

# Parameters: name (str) - Name for the new material; oba (str or bpy.types.Object, optional) - Object to assign material to.
# Usage: Creates a new material and optionally assigns it to an object.
def Mat_Create(name, oba=None):
    """Create a new material."""
    mat = bpy.data.materials.new(name=name)
    mat.use_nodes = True
    if oba:
        obj = Get_Object_Any(oba)
        if obj.type == 'MESH':
            obj.data.materials.append(mat)
        else:
            print(f"Error: Object {obj.name} is not a mesh for material assignment.")
    return mat
##########################################################################

# Parameters: name (str) - Material name; ref (str or bpy.types.Object) - Object to assign material to; slot (int, optional) - Material slot index.
# Usage: Assigns a material to an object’s specified slot.
def Mat_Assign(name, ref, slot=None):
    """Assign a material to an object."""
    obj = Get_Object_Any(ref)
    if obj.type != 'MESH':
        print(f"Error: Object {obj.name} is not a mesh for material assignment.")
        return
    if name in bpy.data.materials:
        mat = bpy.data.materials[name]
        if slot is None:
            obj.data.materials.append(mat)
        else:
            while len(obj.data.materials) <= slot:
                obj.data.materials.append(None)
            obj.data.materials[slot] = mat
    else:
        print(f"Error: Material {name} not found.")
##########################################################################

# Parameters: name (str) - Material name.
# Usage: Returns a material by name.
def Mat_Get(name):
    """Get a material by name."""
    if name in bpy.data.materials:
        return bpy.data.materials[name]
    print(f"Error: Material {name} not found.")
    return None
##########################################################################

# Parameters: name (str) - Material name; oba (str or bpy.types.Object) - Object to assign material to; slot (int) - Material slot index.
# Usage: Assigns a material to an object’s specified slot (alias for Mat_Assign).
def Assign_Material(name, oba, slot=0):
    """Assign a material to an object’s slot."""
    Mat_Assign(name, oba, slot)
##########################################################################

# Parameters: ref (str or bpy.types.Object) - Object to clear materials from.
# Usage: Clears all materials from an object.
def Mat_Clear(ref):
    """Clear all materials from an object."""
    obj = Get_Object_Any(ref)
    if obj.type == 'MESH':
        obj.data.materials.clear()
    else:
        print(f"Error: Object {obj.name} is not a mesh for material clearing.")
##########################################################################

# Parameters: name (str) - Name for the new text object; txt (str) - Text content; col (str or bpy.types.Collection, optional) - Collection to link to.
# Usage: Creates a new text object with specified content and links it to a collection.
def Create_Text(name, txt, col=None):
    """Create a new text object."""
    bpy.ops.object.text_add()
    o = bpy.context.object
    o.name = name
    o.data.body = txt
    if col:
        col_ref = Get_Collection(col) if is_string(col) else col
        if col_ref:
            Move_Object_to_Collection(o, col_ref)
        else:
            print(f"Error: Invalid collection {col} for text object.")
    return o
##########################################################################

# Parameters: None.
# Usage: Converts the active text object to a mesh.
def Text_to_Mesh():
    """Convert active text object to mesh."""
    obj = bpy.context.object
    if obj and obj.type == 'FONT':
        bpy.ops.object.convert(target='MESH')
    else:
        print("Error: No valid text object selected.")
##########################################################################

# Parameters: None.
# Usage: Checks if the active object is a mesh.
def Is_Mesh():
    """Check if active object is a mesh."""
    obj = bpy.context.object
    return 1 if obj and obj.type == 'MESH' else 0
##########################################################################

# Parameters: None.
# Usage: Checks if the active object is a curve.
def Is_Curve():
    """Check if active object is a curve."""
    obj = bpy.context.object
    return 1 if obj and obj.type == 'CURVE' else 0
##########################################################################

# Parameters: None.
# Usage: Checks if the active object is a surface.
def Is_Surface():
    """Check if active object is a surface."""
    obj = bpy.context.object
    return 1 if obj and obj.type == 'SURFACE' else 0
##########################################################################

# Parameters: None.
# Usage: Checks if the active object is a meta object.
def Is_Meta():
    """Check if active object is a meta object."""
    obj = bpy.context.object
    return 1 if obj and obj.type == 'META' else 0
##########################################################################

# Parameters: None.
# Usage: Checks if the active object is a text object.
def Is_Text():
    """Check if active object is a text object."""
    obj = bpy.context.object
    return 1 if obj and obj.type == 'FONT' else 0
##########################################################################

# Parameters: None.
# Usage: Checks if the active object is a grease pencil object.
def Is_GreacePencil():
    """Check if active object is a grease pencil object."""
    obj = bpy.context.object
    return 1 if obj and obj.type == 'GPENCIL' else 0
##########################################################################

# Parameters: None.
# Usage: Checks if the active object is an armature.
def Is_Armature():
    """Check if active object is an armature."""
    obj = bpy.context.object
    return 1 if obj and obj.type == 'ARMATURE' else 0
##########################################################################

# Parameters: None.
# Usage: Checks if the active object is a lattice.
def Is_Lattice():
    """Check if active object is a lattice."""
    obj = bpy.context.object
    return 1 if obj and obj.type == 'LATTICE' else 0
##########################################################################

# Parameters: None.
# Usage: Checks if the active object is an empty.
def Is_Empty():
    """Check if active object is an empty."""
    obj = bpy.context.object
    return 1 if obj and obj.type == 'EMPTY' else 0
##########################################################################

# Parameters: None.
# Usage: Checks if the active object is a light.
def Is_Light():
    """Check if active object is a light."""
    obj = bpy.context.object
    return 1 if obj and obj.type == 'LIGHT' else 0
##########################################################################

# Parameters: None.
# Usage: Checks if the active object is a camera.
def Is_Camera():
    """Check if active object is a camera."""
    obj = bpy.context.object
    return 1 if obj and obj.type == 'CAMERA' else 0
##########################################################################

# Parameters: None.
# Usage: Checks if the active object is a speaker.
def Is_Speaker():
    """Check if active object is a speaker."""
    obj = bpy.context.object
    return 1 if obj and obj.type == 'SPEAKER' else 0
##########################################################################

# Parameters: None.
# Usage: Checks if the active object is a light probe.
def Is_LightProbe():
    """Check if active object is a light probe."""
    obj = bpy.context.object
    return 1 if obj and obj.type == 'LIGHT_PROBE' else 0
##########################################################################

# Parameters: None.
# Usage: Checks if the active object is a point cloud.
def Is_PointCloud():
    """Check if active object is a point cloud."""
    obj = bpy.context.object
    return 1 if obj and obj.type == 'POINTCLOUD' else 0
##########################################################################

# Parameters: None.
# Usage: Checks if the active object is a volume.
def Is_Volume():
    """Check if active object is a volume."""
    obj = bpy.context.object
    return 1 if obj and obj.type == 'VOLUME' else 0
##########################################################################

# Parameters: None.
# Usage: Adds a cube at the 3D cursor location.
def Add_Cube():
    """Add a cube at the cursor location."""
    bpy.ops.mesh.primitive_cube_add()
    return bpy.context.object
##########################################################################

# Parameters: None.
# Usage: Adds a plane at the 3D cursor location.
def Add_Plane():
    """Add a plane at the cursor location."""
    bpy.ops.mesh.primitive_plane_add()
    return bpy.context.object
##########################################################################

# Parameters: None.
# Usage: Adds a UV sphere at the 3D cursor location.
def Add_Sphere():
    """Add a UV sphere at the cursor location."""
    bpy.ops.mesh.primitive_uv_sphere_add()
    return bpy.context.object
##########################################################################

# Parameters: None.
# Usage: Adds a cylinder at the 3D cursor location.
def Add_Cylinder():
    """Add a cylinder at the cursor location."""
    bpy.ops.mesh.primitive_cylinder_add()
    return bpy.context.object
##########################################################################

# Parameters: None.
# Usage: Adds a cone at the 3D cursor location.
def Add_Cone():
    """Add a cone at the cursor location."""
    bpy.ops.mesh.primitive_cone_add()
    return bpy.context.object
##########################################################################

# Parameters: None.
# Usage: Adds a torus at the 3D cursor location.
def Add_Torus():
    """Add a torus at the cursor location."""
    bpy.ops.mesh.primitive_torus_add()
    return bpy.context.object
##########################################################################

# Parameters: None.
# Usage: Adds an icosphere at the 3D cursor location.
def Add_Icosphere():
    """Add an icosphere at the cursor location."""
    bpy.ops.mesh.primitive_ico_sphere_add()
    return bpy.context.object
##########################################################################

# Parameters: None.
# Usage: Deletes selected objects.
def delete_selected_objects():
    """Delete selected objects."""
    try:
        bpy.ops.object.delete()
    except Exception as e:
        print(f"Error deleting selected objects: {str(e)}")
##########################################################################

# Parameters: None.
# Usage: Deselects all objects in the scene.
def Deselect_All():
    """Deselect all objects."""
    for ob in bpy.context.selected_objects:
        ob.select_set(False)
##########################################################################

# Parameters: None.
# Usage: Returns the active object or None if none is active.
def Get_Active_Object():
    """Get the active object."""
    return bpy.context.view_layer.objects.active
##########################################################################

# Parameters: ref (str or bpy.types.Object) - Object to get.
# Usage: Returns an object by name or reference, or the active object if None.
def Get_Object_Any(ref=None):
    """Get an object by name or reference."""
    if ref is None:
        return Get_Active_Object()
    return bpy.data.objects[ref] if is_string(ref) else ref
##########################################################################

# Parameters: None.
# Usage: Returns the last selected object or None if none selected.
def Get_Last_Object():
    """Get the last selected object."""
    selected = bpy.context.selected_objects
    return selected[-1] if selected else None
##########################################################################

# Parameters: None.
# Usage: Returns the current keyframe number.
def Get_Actual_KF():
    """Get the current keyframe."""
    return bpy.context.scene.frame_current
##########################################################################

# Parameters: None.
# Usage: Saves the current blend file.
def SaveFile():
    """Save the current blend file."""
    try:
        bpy.ops.wm.save_mainfile()
    except Exception as e:
        print(f"Error saving file: {str(e)}")
##########################################################################

# Parameters: None.
# Usage: Reloads Python modules and clears garbage collection to reset the Python environment.
def Reset_Python():
    """Reload Python modules and clear garbage collection."""
    print("Resetting Python environment...")
    for mod in list(sys.modules.values()):
        try:
            importlib.reload(mod)
        except Exception as e:
            print(f"Error reloading module {mod}: {str(e)}")
    gc.collect()
    print("Python reset complete.")
##########################################################################

# Parameters: None.
# Usage: Returns the selected objects as a list, with the active object first.
def Remember_Selected():
    """Remember the selected objects."""
    act = Get_Active_Object()
    asl = bpy.context.selected_objects
    asl.insert(0, act)
    return asl
##########################################################################

# Parameters: i (list) - List of objects to restore selection for.
# Usage: Restores the selection state from a saved list.
def Restore_Selected(i):
    """Restore selection from a saved list."""
    Deselect_All()
    act = i[0]
    for obj in i:
        Select_Object(obj)
    Set_Active_Object(act)
##########################################################################

# Parameters: img (str) - Image name; name (str) - Prefix for created objects; col (str or bpy.types.Collection, optional) - Collection to link objects to.
# Usage: Creates objects from an image’s pixels, placing cubes at non-transparent pixels.
def Image_to_Objects(img, name, col=None):
    """Create objects from image pixels."""
    if img not in bpy.data.images:
        print(f"Error: Image {img} not found.")
        return
    image = bpy.data.images[img]
    pixels = list(image.pixels)
    width = image.size[0]
    height = image.size[1]
    col_ref = Get_Collection(col) if col else bpy.context.view_layer.active_layer_collection.collection
    for y in range(height):
        for x in range(width):
            i = (y * width + x) * 4
            r, g, b, a = pixels[i:i+4]
            if a > 0.0:
                o = Create_Object(f"{name}_{x}_{y}", col_ref)
                o.location = (x, y, 0)
                bpy.ops.mesh.primitive_cube_add(size=1.0, location=(x, y, 0))
                o.data = bpy.context.object.data
                delete_selected_objects()
                Link_Object_to_Collection(o, col_ref)
    print(f"Created objects from image {img}.")
##########################################################################

# Parameters: vis (bool) - Visibility state; fr (int) - Frame number; ta (int) - Transition type.
# Usage: Sets visibility keyframe for selected objects at the specified frame.
def Ani_View_Selected_Objects_at_KF(vis, fr, ta=0):
    """Set visibility keyframe for selected objects."""
    for obj in bpy.context.selected_objects:
        obj.hide_viewport = not vis
        obj.hide_render = not vis
        obj.keyframe_insert(data_path="hide_viewport", frame=fr)
        obj.keyframe_insert(data_path="hide_render", frame=fr)
        if ta == 1:
            obj.hide_viewport = vis
            obj.hide_render = vis
            obj.keyframe_insert(data_path="hide_viewport", frame=(fr + 1))
            obj.keyframe_insert(data_path="hide_render", frame=(fr + 1))
            obj.hide_viewport = not vis
            obj.hide_render = not vis
            obj.keyframe_insert(data_path="hide_viewport", frame=(fr - 1))
            obj.keyframe_insert(data_path="hide_render", frame=(fr - 1))
    print(f"Set visibility keyframe at frame {fr} for selected objects.")
##########################################################################

# Parameters: None.
# Usage: Enables the Cell Fracture add-on if available.
def enable_cell_fracture():
    """Enable Cell Fracture add-on."""
    try:
        bpy.ops.preferences.addon_enable(module="object_fracture_cell")
        print("Cell Fracture add-on enabled.")
    except Exception as e:
        print(f"Error enabling Cell Fracture add-on: {str(e)}")
##########################################################################
##########################################################################
# Parameters: source_obj (str or bpy.types.Object) - Source object to fracture; num_cells (int) - Number of fracture cells.
# Usage: Fractures the selected object using Cell Fracture add-on.
def Cell_Fracture_Selected(source_obj, num_cells=10):
    """Fracture selected object using Cell Fracture.
    Warning: Requires Cell Fracture add-on.
    """
    obj = Get_Object_Any(source_obj)
    if obj.type != 'MESH':
        print(f"Error: Object {obj.name} is not a mesh for fracturing.")
        return
    try:
        enable_cell_fracture()
        bpy.ops.object.select_all(action='DESELECT')
        obj.select_set(True)
        bpy.context.view_layer.objects.active = obj
        bpy.ops.object.add_fracture_cell_objects(cell_count=num_cells)
        print(f"Fractured {obj.name} into {num_cells} cells.")
    except Exception as e:
        print(f"Error fracturing object: {str(e)}")
##########################################################################
##########################################################################

# Parameters: source_obj (str or bpy.types.Object) - Source object to fracture.
# Usage: Fractures the selected object using Cell Fracture add-on with default settings.
def Cell_Fracture_Selected_A(source_obj):
    """Fracture selected object using Cell Fracture with defaults.
    Warning: Requires Cell Fracture add-on.
    """
    Cell_Fracture_Selected(source_obj, num_cells=10)
##########################################################################

# Parameters: None.
# Usage: Converts selected curves to meshes.
def Curve_to_Mesh():
    """Convert selected curves to meshes."""
    for obj in bpy.context.selected_objects:
        if obj.type == 'CURVE':
            bpy.context.view_layer.objects.active = obj
            bpy.ops.object.convert(target='MESH')
    print("Converted selected curves to meshes.")
##########################################################################

# Parameters: None.
# Usage: Joins selected objects into a single mesh.
def Join_selected():
    """Join selected objects into a single mesh."""
    if len(bpy.context.selected_objects) < 2:
        print("Error: Select at least two objects to join.")
        return
    try:
        bpy.ops.object.join()
        print("Joined selected objects.")
    except Exception as e:
        print(f"Error joining objects: {str(e)}")
##########################################################################

# Parameters: name (str) - Name for the new empty object; col (str or bpy.types.Collection, optional) - Collection to link to.
# Usage: Creates a new empty object and links it to a collection.
def Create_Empty(name, col=None):
    """Create a new empty object."""
    o = bpy.data.objects.new(name, None)
    col_ref = Get_Collection(col) if col else bpy.context.view_layer.active_layer_collection.collection
    col_ref.objects.link(o)
    return o
##########################################################################

# Parameters: name (str) - Name for the new camera; col (str or bpy.types.Collection, optional) - Collection to link to.
# Usage: Creates a new camera and links it to a collection.
def Create_Camera(name, col=None):
    """Create a new camera."""
    cam_data = bpy.data.cameras.new(name)
    o = bpy.data.objects.new(name, cam_data)
    col_ref = Get_Collection(col) if col else bpy.context.view_layer.active_layer_collection.collection
    col_ref.objects.link(o)
    return o
##########################################################################

# Parameters: name (str) - Name for the new light; typ (str) - Light type ('POINT', 'SUN', etc.); col (str or bpy.types.Collection, optional) - Collection to link to.
# Usage: Creates a new light with the specified type and links it to a collection.
def Create_Light(name, typ='POINT', col=None):
    """Create a new light."""
    light_data = bpy.data.lights.new(name, type=typ)
    o = bpy.data.objects.new(name, light_data)
    col_ref = Get_Collection(col) if col else bpy.context.view_layer.active_layer_collection.collection
    col_ref.objects.link(o)
    return o
##########################################################################

# Parameters: None.
# Usage: Starts a performance timer and returns the start time.
def Start_Timer():
    """Start a performance timer."""
    return time.time()
##########################################################################

# Parameters: t (float) - Start time from Start_Timer().
# Usage: Ends the timer and prints the elapsed time.
def End_Timer(t):
    """End a performance timer and print elapsed time."""
    elapsed = time.time() - t
    print(f"Elapsed time: {elapsed:.2f} seconds")
##########################################################################

# Parameters: None.
# Usage: Converts selected meshes to curves.
def Mesh_to_Curve():
    """Convert selected meshes to curves."""
    for obj in bpy.context.selected_objects:
        if obj.type == 'MESH':
            bpy.context.view_layer.objects.active = obj
            bpy.ops.object.convert(target='CURVE')
    print("Converted selected meshes to curves.")
##########################################################################

# Parameters: None.
# Usage: Converts selected meshes to grease pencil objects.
def Mesh_to_GreacePencil():
    """Convert selected meshes to grease pencil objects."""
    for obj in bpy.context.selected_objects:
        if obj.type == 'MESH':
            bpy.context.view_layer.objects.active = obj
            bpy.ops.object.convert(target='GPENCIL')
    print("Converted selected meshes to grease pencil.")
##########################################################################

def Trans_Location(oba=None, loc=None):
    """
    Sets or gets the location of an object as a Vector.
    
    Parameters:
    - oba (str or bpy.types.Object, optional): Object name or reference. Defaults to active object.
    - loc (list/tuple of 3 floats, optional): New location [x, y, z]. If None, returns current location.
    
    Returns:
    - mathutils.Vector: Current or set location.
    
    Usage:
    >>> Trans_Location('Cube', [1.0, 2.0, 3.0])  # Set Cube location
    >>> current_loc = Trans_Location()  # Get active object location
    
    Features: Validates input, updates view layer, handles non-mesh objects.
    """
    obj = Get_Object_Any(oba)  # Assumes Get_Object_Any from new library
    if obj is None:
        print("Error: No valid object provided or found.")
        return None
    
    obj_provided = oba is not None
    loc_provided = loc is not None
    
    if obj_provided:
        if loc_provided:
            # Validate loc as 3-element sequence
            if len(loc) != 3 or not all(isinstance(v, (int, float)) for v in loc):
                print("Warning: Invalid location format. Expected [x, y, z] floats.")
                return obj.location
            obj.location = Vector((loc[0], loc[1], loc[2]))
            bpy.context.view_layer.update()  # Ensure viewport refresh
            return obj.location
        else:
            return obj.location
    else:
        if loc_provided:
            obj = Get_Active_Object()  # Fallback to active
            if obj:
                obj.location = Vector((loc[0], loc[1], loc[2]))
                bpy.context.view_layer.update()
                return obj.location
            else:
                print("Error: No active object to set location.")
                return None
        else:
            obj = Get_Active_Object()
            return obj.location if obj else None
##########################################################################
def Trans_Rotation(oba=None, rot=None):
    """
    Sets or gets the Euler rotation of an object.
    
    Parameters:
    - oba (str or bpy.types.Object, optional): Object name or reference. Defaults to active object.
    - rot (list/tuple of 3 floats, optional): New rotation in radians [x, y, z]. If None, returns current rotation.
    
    Returns:
    - mathutils.Euler or None: Current or set rotation.
    
    Usage:
    >>> Trans_Rotation('Cube', [0, 0, 1.57])  # Rotate Cube 90 degrees on Z
    >>> current_rot = Trans_Rotation()  # Get active object rotation
    
    Features: Ensures 'XYZ' rotation mode, validates radians input, auto-updates.
    """
    obj = Get_Object_Any(oba)  # Assumes Get_Object_Any from new library
    if obj is None:
        print("Error: No valid object provided or found.")
        return None
    
    # Ensure rotation mode is Euler XYZ for compatibility
    if obj.rotation_mode != 'XYZ':
        obj.rotation_mode = 'XYZ'
    
    obj_provided = oba is not None
    rot_provided = rot is not None
    
    if obj_provided:
        if rot_provided:
            # Validate rot as 3-element sequence
            if len(rot) != 3 or not all(isinstance(v, (int, float)) for v in rot):
                print("Warning: Invalid rotation format. Expected [x, y, z] radians.")
                return obj.rotation_euler
            obj.rotation_euler = Euler((rot[0], rot[1], rot[2]))
            bpy.context.view_layer.update()
            return obj.rotation_euler
        else:
            return obj.rotation_euler
    else:
        if rot_provided:
            obj = Get_Active_Object()
            if obj:
                if obj.rotation_mode != 'XYZ':
                    obj.rotation_mode = 'XYZ'
                obj.rotation_euler = Euler((rot[0], rot[1], rot[2]))
                bpy.context.view_layer.update()
                return obj.rotation_euler
            else:
                print("Error: No active object to set rotation.")
                return None
        else:
            obj = Get_Active_Object()
            return obj.rotation_euler if obj else None
   
##########################################################################
def Make_Vector(data):
    """
    Converts a list/tuple to a mathutils.Vector.
    
    Parameters:
    - data (list/tuple of 3 floats/ints): Input data [x, y, z].
    
    Returns:
    - mathutils.Vector: Constructed vector.
    
    Usage:
    >>> vec = Make_Vector([1.0, 2.0, 3.0])
    
    Features: Validates length and types, falls back to zero vector on error.
    """
    if not isinstance(data, (list, tuple)) or len(data) != 3:
        print("Warning: Invalid data for Vector. Expected 3-element list/tuple.")
        return Vector((0.0, 0.0, 0.0))
    try:
        return Vector((float(data[0]), float(data[1]), float(data[2])))
    except (ValueError, TypeError):
        print("Warning: Could not convert data to floats. Using zero vector.")
        return Vector((0.0, 0.0, 0.0))
##########################################################################
def Clear_unused_Data(a=1):
    """
    Purges unused data and removes specific collections like RigidBodyWorld.
    
    Parameters:
    - a (int, optional): Flag for RigidBodyWorld removal (1 = remove).
    
    Returns:
    - None
    
    Usage:
    >>> Clear_unused_Data()
    
    Features: Scans for orphans, handles non-existent collections gracefully.
    """
    if a == 1:
        if "RigidBodyWorld" in bpy.data.collections:
            bpy.data.collections.remove(bpy.data.collections["RigidBodyWorld"])
            print("Removed RigidBodyWorld collection.")
    
    try:
        bpy.ops.outliner.orphans_purge()
        print("Purged unused data (meshes, materials, etc.).")
    except Exception as e:
        print(f"Error purging orphans: {str(e)}")
    
    bpy.context.view_layer.update()  # Refresh scene


##########################################################################
def Duplicate_Object_with_name(name='Cube', newname='Cube_001'):
    """
    Creates a full independent copy of an object, clearing modifiers/constraints.
    
    Parameters:
    - name (str): Source object name.
    - newname (str): New object name.
    
    Returns:
    - bpy.types.Object or None: New object.
    
    Usage:
    >>> new_obj = Duplicate_Object_with_name('Cube', 'Cube_Copy')
    
    Features: Copies data, clears animation/modifiers/constraints, links to scene.
    """
    if name not in bpy.data.objects:
        print(f"Error: Object '{name}' not found.")
        return None
    
    obj = bpy.data.objects[name]
    new_obj = bpy.data.objects.new(newname, obj.data.copy() if obj.data else None)
    new_obj.animation_data_clear()  # Clear any linked animation
    
    # Clear modifiers and constraints for independence
    new_obj.modifiers.clear()
    new_obj.constraints.clear()
    
    # Link to scene collection
    bpy.context.scene.collection.objects.link(new_obj)
    
    print(f"Duplicated '{name}' to '{newname}'.")
    bpy.context.view_layer.update()
    return new_obj

##########################################################################

def Duplicate_Object(obj, newname='Cube_001'):
    """
    Wrapper for duplicating an object by reference or name.
    
    Parameters:
    - obj (str or bpy.types.Object): Source object.
    - newname (str): New object name.
    
    Returns:
    - bpy.types.Object or None: New object.
    
    Usage:
    >>> new_obj = Duplicate_Object('Cube', 'Cube_Copy')
    
    Features: Auto-converts string to object ref.
    """
    src_name = obj.name if hasattr(obj, 'name') else obj
    return Duplicate_Object_with_name(src_name, newname)


##########################################################################

def Make_Linked(original_obj, lnk=0):
    """
    Creates a linked or full copy of an object (internal helper for planks/lines).
    
    Parameters:
    - original_obj (str or bpy.types.Object): Source.
    - lnk (int): 0=full copy, 1=linked duplicate.
    
    Returns:
    - bpy.types.Object: New object.
    
    Usage: Internal use.
    """
    src = Get_Object_Any(original_obj)
    if not src:
        print("Error: Invalid source for Make_Linked.")
        return None
    if lnk == 0:
        return Copy_Object(src)  # Assumes Copy_Object from new library
    else:
        new_obj = src.copy()
        new_obj.data = src.data  # Linked data
        bpy.context.scene.collection.objects.link(new_obj)
        return new_obj

##########################################################################

def MakeNewPlank(col, Plank, xpo, ypo, zpo, rox, lnk=0):
    """
    Creates a positioned/rotated plank (linked or copy) in a collection.
    
    Parameters:
    - col (str or bpy.types.Collection): Target collection.
    - Plank (str or bpy.types.Object): Source plank object.
    - xpo, ypo, zpo (float): Position.
    - rox (float): Z rotation in radians.
    - lnk (int): 0=full copy, 1=linked.
    
    Returns:
    - bpy.types.Object: New plank.
    
    Usage:
    >>> plank = MakeNewPlank('Planks', 'TemplatePlank', 0, 0, 0, 0.0)
    """
    obj = Make_Linked(Plank, lnk)
    if not obj:
        return None
    MoveObjToCollection(col, obj)  # Ported below
    Trans_Location(obj, [xpo, ypo, zpo])
    obj.rotation_mode = 'XYZ'
    Trans_Rotation(obj, [0, 0, rox])
    return obj

##########################################################################
"""
    Duplicates selected objects in a line along specified axis.
    
    Parameters:
    - inp (int): Number of duplicates.
    - dir (int): 0=X, 1=Y, 2=Z.
    - dist (float): Spacing multiplier.
    - lnk (int): 0=full copy, 1=linked.
    - scene_update (bool, new): Refresh viewport.
    
    Returns:
    - None
    
    Usage:
    >>> Makeline(5, dir=0)  # Line along X
"""
def Makeline(inp=3, dir=0, dist=1.01, lnk=0, scene_update=True):
    olist = bpy.context.selected_objects[:]
    if not olist:
        print("Error: No objects selected for Makeline.")
        return
    nam = "Dups_" + str(dir) if lnk == 0 else "RealObj"
    col = NewCollection(nam)
    tar = inp + 1
    for num in range(1, tar):
        print(f"Lining Up {dir}: {num}/{tar} lnk:{lnk}")
        for oba in olist:
            size_vec = Geo_Size_Vect(dir)  # Assumes Geo_Size_Vect exists or port if needed; fallback to unit
            if 'Geo_Size_Vect' not in globals():
                size_vec = 1.0  # Fallback
            offset = size_vec * dist * num
            obj = Make_Linked(oba, lnk)
            MoveObjToCollection(col, obj)
            loc = list(obj.location)
            if dir == 0:
                loc[0] += offset
            elif dir == 1:
                loc[1] += offset
            else:
                loc[2] += offset
            Trans_Location(obj, loc)
    if scene_update:
        bpy.context.view_layer.update()

##########################################################################
"""X-axis line (wrapper)."""
def MakelineX(inp=3, dist=1.01, lnk=0, scene_update=True):    
    Makeline(inp, 0, dist, lnk, scene_update)

##########################################################################
"""Y-axis line (wrapper)."""
def MakelineY(inp=3, dist=1.01, lnk=0, scene_update=True):    
    Makeline(inp, 1, dist, lnk, scene_update)

##########################################################################
"""Z-axis line (wrapper)."""
def MakelineZ(inp=3, dist=1.01, lnk=0, scene_update=True):
    Makeline(inp, 2, dist, lnk, scene_update)

##########################################################################
""" Creates XY quad grid. """
def MakeQuadXY(num=3, dist=1.01, lnk=0, scene_update=True):    
    num -= 1
    MakelineX(num, dist, lnk, False)
    MakelineY(num, dist, lnk, scene_update)
##########################################################################
def MakeQuadXZ(num=3, dist=1.01, lnk=0, scene_update=True):
    """
    Creates XZ quad grid.
    """
    num -= 1
    MakelineX(num, dist, lnk, False)
    MakelineZ(num, dist, lnk, scene_update)
##########################################################################
def MakeQuadYZ(num=3, dist=1.01, lnk=0, scene_update=True):
    """
    Creates YZ quad grid.
    """
    num -= 1
    MakelineY(num, dist, lnk, False)
    MakelineZ(num, dist, lnk, scene_update)
##########################################################################
def MakeCube(num=3, hi=3, lnk=0, dist=1.01, scene_update=True):
    """
    Builds 3D cube grid.
    """
    print("MakeCube Phase 1")
    MakeQuadXY(num, dist, lnk, False)
    hi -= 1
    print("MakeCube Phase 2")
    MakelineZ(hi, dist, lnk, scene_update)
##########################################################################

def PlaceInCube(CubeSize=10, scene_update=True):
    """
    Randomly places selected objects in a cube volume.
    
    Parameters:
    - CubeSize (float): Half-side length of cube.
    - scene_update (bool, new): Refresh viewport.
    
    Returns:
    - None
    
    Usage:
    >>> PlaceInCube(5.0)
    """
    d = CubeSize / 2
    selected = bpy.context.selected_objects
    if not selected:
        print("Error: No objects selected for PlaceInCube.")
        return
    for obj in selected:
        xpos = r.uniform(-d, d)
        ypos = r.uniform(-d, d)
        zpos = r.uniform(-d, d)
        obj.rotation_euler = (0, 0, 0)
        Trans_Location(obj, [xpos, ypos, zpos])
    print("Placed objects in cube.")
    if scene_update:
        bpy.context.view_layer.update()
##########################################################################
def MakeQuad(col, Plank, si, cor, x, y, z, lnk=0, scene_update=True):
    """
    Builds polygonal quad layer.
    """
    phi = 0.9
    pi = 3.1415926
    pi2 = 2 * pi
    con = 180 / pi
    step = pi2 / cor
    wi = (360 / cor) / con
    i = 0
    ro = 0
    target = pi2 - 0.1
    while i < target:
        if z % 2 == 1:  # Bitwise & for odd/even
            rox = 43 * m.pi / 180  # Degrees to radians
            row = 1
        else:
            rox = 0
            row = 0
        xp = m.sin(i + rox) * si + x
        yp = m.cos(i + rox) * si + y
        zp = z * (2 * phi) - phi
        MakeNewPlank(col, Plank, xp, yp, zp, ro + row, lnk)
        ro += wi
        i += step
    if scene_update:
        bpy.context.view_layer.update()

##########################################################################
def MakeQuadTower(Tower_high=80, cor=8, x=0, y=0, z=0, lnk=0, scene_update=True):
    """
    Builds multi-floor quad tower.
    """
    import time
    then = time.time()
    Plank_Len = 12
    siz = 0.8
    hg = 0.9
    Add_Cube()  # Assumes from new library; adds at origin
    obj = bpy.context.object
    # cube2plank not in old snippet; assuming scales/cuts cube to plank
    # Placeholder: obj.scale = (siz, Plank_Len, hg)
    obj.scale = (siz, Plank_Len, hg)
    # ReCenterGeo placeholder: Set origin to geometry
    bpy.ops.object.origin_set(type='ORIGIN_GEOMETRY', center='MEDIAN')
    myobj = ReCenterGeo(obj, hg) if 'ReCenterGeo' in globals() else obj  # Fallback
    col = NewCollection("Quad")
    LinkToCollection(col, obj)
    si = Plank_Len + (siz * 2.6)
    for i in range(1, Tower_high):
        print(f"Construction ongoing: {i}")
        MakeQuad(col, myobj, si, cor, x, y, i, lnk, False)
    now = time.time()
    print(f"Built Quad-Tower in: {now - then} seconds.")
    if scene_update:
        bpy.context.view_layer.update()


##########################################################################
import math as m

def MakeCircle_S(col, Plank, ap, xof, yof, y, lnk=0, scene_update=True):
    """
    Builds rotated circular support layer.
    """
    pi = 3.14159267
    pi2 = 2 * pi
    tar = pi2 - 0.001
    sz = 0.8
    phi = 0.9
    Plank_Len = 6
    ps = 6
    i = 0
    sg = 1 - sz
    Anz_Planks = ap * 4
    inc = pi2 / Anz_Planks
    ai = 360 / Anz_Planks
    roi = -ai / ((ps * 10.05) - (ps / 2))
    s = Anz_Planks * pi * 0.65
    s = (s / 100) * (49 / 2)
    ro = 0
    cnt = 0
    while i < tar:
        i += inc
        cnt += 1
        dx = m.sin(i) * s + xof
        dy = m.cos(i) * s + yof
        dz = y * (2 * phi) - phi
        ro += roi
        rox = ro + 29.8 * m.pi / 180  # To radians
        MakeNewPlank(col, Plank, dx, dy, dz, rox, lnk)
    if scene_update:
        bpy.context.view_layer.update()
##########################################################################

def MakeCircle_L(col, Plank, ap, xof, yof, y, sig, lnk=0, scene_update=True):
    """
    Builds non-rotated circular load layer.
    """
    pi = 3.1415926
    pi2 = 2 * pi
    tar = pi2 - 0.001
    sz = 0.8
    phi = 0.9
    Plank_Len = 6
    ps = 6
    i = 0
    sg = 1 - sz
    Anz_Planks = ap
    ma = Anz_Planks + 1
    inc = pi2 / Anz_Planks
    ai = 360 / Anz_Planks
    roi = -ai / ((ps * 10.05) - (ps / 2))
    saf = (1 / Anz_Planks) * 0.25
    sif = 0.334 - saf if sig == 1 else 0.337 + saf
    s = Anz_Planks * pi2 * sif
    ro = 0
    cnt = 0
    while i < tar:
        i += inc
        cnt += 1
        dx = m.sin(i) * s + xof
        dy = m.cos(i) * s + yof
        dz = y * (2 * phi) - phi
        ro += roi
        rox = ro + 0.3 * m.pi / 180
        MakeNewPlank(col, Plank, dx, dy, dz, rox, lnk)
    if scene_update:
        bpy.context.view_layer.update()
##########################################################################

def Roundtower(x, y, size, hi, lnk=0, scene_update=True):
    """
    Builds alternating round tower.
    """
    import time
    then = time.time()
    Plank_Len = 6
    siz = 0.8
    hg = 0.9
    Add_Cube()
    obj = bpy.context.object
    obj.scale = (siz, Plank_Len, hg)
    myobj = ReCenterGeo(obj, hg) if 'ReCenterGeo' in globals() else obj
    Trans_Location(myobj, [0, 0, -10])
    col = NewCollection("Tower")
    LinkToCollection(col, obj)
    for za in range(1, hi):
        print(f"Building Level: {za}")
        ypos = y
        xpos = x
        SizeA = size
        SizeB = size - (size / 100 * 5)
        if za % 2 == 1:
            MakeCircle_S(col, myobj, SizeA, xpos, ypos, za, lnk, False)
        else:
            MakeCircle_L(col, myobj, SizeB, xpos, ypos, za, 1, lnk, False)
            MakeCircle_L(col, myobj, SizeB, xpos, ypos, za, 0, lnk, False)
    now = time.time()
    print(f"It took: {now - then} seconds")
    if scene_update:
        bpy.context.view_layer.update()
    return za

##########################################################################
import bpy
import random as r

def RB_MakeSceneA(scene_update=True):
    """
    Generates chain-like rigid body scene.
    """
    bpy.ops.mesh.primitive_plane_add(size=440, location=(0, 0, 0))
    bpy.ops.rigidbody.object_add()
    bpy.context.object.rigid_body.type = 'PASSIVE'
    for x in range(1, 19):
        bpy.ops.mesh.primitive_torus_add(location=(0, x*4.3, 110), rotation=(0, 1.5708*(x%2), 0), major_radius=3.5, minor_radius=0.5, abso_major_rad=1.25, abso_minor_rad=0.75)
        bpy.ops.rigidbody.object_add()
        bpy.context.object.rigid_body.collision_shape = 'MESH'
        print(f"Generating Chains: {x}/19")
        if x == 1:
            bpy.context.object.rigid_body.enabled = False
        for z in range(0, 9):
            print(f"Generating Cubes: {z}/9")
            for m in range(0, 9):
                bpy.ops.mesh.primitive_cube_add(size=5.9, location=(x*6-60, 2+m*6, 2.8+z*6))
                bpy.ops.rigidbody.object_add()
                bpy.context.object.rigid_body.mass = 0.0001
    if scene_update:
        bpy.context.view_layer.update()

##########################################################################

def RB_MakeSceneB(lnk=0, scene_update=True):
    """
    Generates city-like tower scene with RB.
    """
    print("This may take a while. Please be patient.")
    Add_Plane(10, 10, 0, 2580)  # Assumes Add_Plane with params from new lib
    for i in range(1, 10):
        print(f"Starting up: {i}/10")
        pow = 0.013 + ((10 - i) * 0.02)
        if i == 2:
            dea = 0
        else:
            dea = 1
        Hi = 40
        x = 0
        y = 0
        szi = 10
        for j in range(1, szi):
            for i in range(1, szi):  # Nested i, but ok
                h = r.randint(80, 160)
                t = r.randint(0, 2)
                k = 4 if t > 1 else 8
                print(f"Building Towers: {j}/{szi}")
                MakeQuadTower(h + x, k + y, i * 80, j * 80, 0, lnk, False)
    # Bake placeholder: Assumes RB_Bake exists
    if 'RB_Bake' in globals():
        RB_Bake(240, 240)
    if scene_update:
        bpy.context.view_layer.update()


##########################################################################

def Text_Create(textname):
    """
    Creates a new text datablock.
    
    Parameters:
    - textname (str): Name for text.
    
    Returns:
    - bpy.types.Text: New text.
    """
    if textname in bpy.data.texts:
        print(f"Warning: Text '{textname}' exists; overwriting.")
    t = bpy.data.texts.new(textname)
    return t
##########################################################################

def Text_Object_Delete(textname):
    """
    Deletes a text datablock.
    """
    if is_string(textname):
        if textname in bpy.data.texts:
            t = bpy.data.texts[textname]
            bpy.data.texts.remove(t)
            print(f"Deleted text '{textname}'.")
        else:
            print(f"Error: Text '{textname}' not found.")
    else:
        bpy.data.texts.remove(textname)
        print("Deleted text object.")
##########################################################################

def Text_Object_get_Lines(textname):
    """
    Gets lines from a text datablock.
    """
    if textname in bpy.data.texts:
        return bpy.data.texts[textname].lines
    else:
        print(f"Error: Text '{textname}' not found.")
        return []


##########################################################################
def Link_Objects_to_Collection(ref, col):
    """
    Links list of objects to collection.
    """
    col_ref = Get_Collection(col) if is_string(col) else col
    if not col_ref:
        print("Error: Invalid collection.")
        return
    for o in ref:
        col_ref.objects.link(o)
    bpy.context.view_layer.update()

##########################################################################

def Unlink_Object_from_Collection(ref, col):
    """
    Unlinks single object from collection.
    """
    col_ref = Get_Collection(col) if is_string(col) else col
    obj_ref = Get_Object_Any(ref) if is_string(ref) else ref
    if col_ref and obj_ref:
        col_ref.objects.unlink(obj_ref)

##########################################################################

def Unlink_Objects_from_Collection(lis, col):
    """
    Unlinks list of objects from collection.
    """
    col_ref = Get_Collection(col) if is_string(col) else col
    if not col_ref:
        print("Error: Invalid collection.")
        return
    for o in lis:
        col_ref.objects.unlink(o)
    bpy.context.view_layer.update()

##########################################################################

def Move_Object_to_Collection(ref, col):
    """
    Moves single object to new collection.
    """
    obj_ref = Get_Object_Any(ref) if is_string(ref) else ref
    col_ref = Get_Collection(col) if is_string(col) else col
    if not col_ref or not obj_ref:
        print("Error: Invalid object or collection.")
        return
    for c in obj_ref.users_collection:
        c.objects.unlink(obj_ref)
    Link_Object_to_Collection(obj_ref, col_ref)  # Assumes from new lib
    bpy.context.view_layer.update()

##########################################################################

def Move_Objects_to_Collection(ref, col):
    """
    Moves list of objects to new collection.
    """
    col_ref = Get_Collection(col) if is_string(col) else col
    if not col_ref:
        print("Error: Invalid collection.")
        return
    for o in ref:
        for c in o.users_collection:
            c.objects.unlink(o)
        Link_Object_to_Collection(o, col_ref)
    bpy.context.view_layer.update()

##########################################################################

def Get_Object_Any_collection(ref):
    """
    Gets primary collection for object.
    """
    obj_ref = Get_Object_Any(ref) if is_string(ref) else ref
    if obj_ref and obj_ref.users_collection:
        return obj_ref.users_collection[0]
    return None

##########################################################################

def Get_Object_Any_collections(ref):
    """
    Gets all collections for object.
    """
    obj_ref = Get_Object_Any(ref) if is_string(ref) else ref
    return obj_ref.users_collection if obj_ref else []

##########################################################################
def Mod_add(ref, modname, mod_type):
    """
    Generic modifier adder (internal).
    """
    obj = Get_Object_Any(ref)
    if not obj or obj.type != 'MESH':
        print("Error: Invalid mesh for modifier.")
        return None
    try:
        bpy.ops.object.modifier_add(type=mod_type)
        mod = obj.modifiers[-1]
        mod.name = modname
        return mod
    except Exception as e:
        print(f"Error adding {mod_type}: {str(e)}")
        return None

##########################################################################
# Generate all add_* wrappers (non-shortened: all listed)
def add_data_transfer(ref, modname="DataTransfer"):
    return Mod_add(ref, modname, 'DATA_TRANSFER')

def add_mesh_cache(ref, modname="MeshCache"):
    return Mod_add(ref, modname, 'MESH_CACHE')

def add_mesh_sequence_cache(ref, modname="MeshSequenceCache"):
    return Mod_add(ref, modname, 'MESH_SEQUENCE_CACHE')

def add_normal_edit(ref, modname="NormalEdit"):
    return Mod_add(ref, modname, 'NORMAL_EDIT')

def add_weighted_normal(ref, modname="WeightedNormal"):
    return Mod_add(ref, modname, 'WEIGHTED_NORMAL')

def add_uv_project(ref, modname="UVProject"):
    return Mod_add(ref, modname, 'UV_PROJECT')

def add_uv_warp(ref, modname="Warp"):
    return Mod_add(ref, modname, 'UV_WARP')

def add_vertex_weight_edit(ref, modname="VertexWeightEdit"):
    return Mod_add(ref, modname, 'VERTEX_WEIGHT_EDIT')

def add_vertex_weight_mix(ref, modname="VertexWeightMix"):
    return Mod_add(ref, modname, 'VERTEX_WEIGHT_MIX')

def add_vertex_weight_proximity(ref, modname="VertexWeightProximity"):
    return Mod_add(ref, modname, 'VERTEX_WEIGHT_PROXIMITY')

def add_array(ref, modname="Array"):
    return Mod_add(ref, modname, 'ARRAY')

def add_bevel(ref, modname="Bevel"):
    return Mod_add(ref, modname, 'BEVEL')

def add_boolean(ref, modname="Boolean"):
    return Mod_add(ref, modname, 'BOOLEAN')

def add_build(ref, modname="Build"):
    return Mod_add(ref, modname, 'BUILD')

def add_decimate(ref, modname="Decimate"):
    return Mod_add(ref, modname, 'DECIMATE')

def add_edge_split(ref, modname="EdgeSplit"):
    return Mod_add(ref, modname, 'EDGE_SPLIT')

def add_mask(ref, modname="Mask"):
    return Mod_add(ref, modname, 'MASK')

def add_mirror(ref, modname="Mirror"):
    return Mod_add(ref, modname, 'MIRROR')

def add_multires(ref, modname="Multires"):
    return Mod_add(ref, modname, 'MULTIRES')

def add_remesh(ref, modname="Remesh"):
    return Mod_add(ref, modname, 'REMESH')

def add_screw(ref, modname="Screw"):
    return Mod_add(ref, modname, 'SCREW')

def add_skin(ref, modname="Skin"):
    return Mod_add(ref, modname, 'SKIN')

def add_solidify(ref, modname="Solidify"):
    return Mod_add(ref, modname, 'SOLIDIFY')

def add_subsurf(ref, modname="Subsurf"):
    return Mod_add(ref, modname, 'SUBSURF')

def add_triangulate(ref, modname="Triangulate"):
    return Mod_add(ref, modname, 'TRIANGULATE')

def add_weld(ref, modname="Weld"):
    return Mod_add(ref, modname, 'WELD')

def add_wireframe(ref, modname="Wireframe"):
    return Mod_add(ref, modname, 'WIREFRAME')

def add_armature(ref, modname="Armature"):
    return Mod_add(ref, modname, 'ARMATURE')

def add_cast(ref, modname="Cast"):
    return Mod_add(ref, modname, 'CAST')

def add_curve(ref, modname="Curve"):
    return Mod_add(ref, modname, 'CURVE')

def add_displace(ref, modname="Displace"):
    return Mod_add(ref, modname, 'DISPLACE')

def add_hook(ref, modname="Hook"):
    return Mod_add(ref, modname, 'HOOK')

def add_laplacian_deform(ref, modname="LaplacianDeform"):
    return Mod_add(ref, modname, 'LAPLACIANDEFORM')

def add_lattice(ref, modname="Lattice"):
    return Mod_add(ref, modname, 'LATTICE')

def add_mesh_deform(ref, modname="Deform"):
    return Mod_add(ref, modname, 'MESH_DEFORM')

def add_shrinkwrap(ref, modname="Shrinkwrap"):
    return Mod_add(ref, modname, 'SHRINKWRAP')

def add_simple_deform(ref, modname="SimpleDeform"):
    return Mod_add(ref, modname, 'SIMPLE_DEFORM')

def add_smooth(ref, modname="Smooth"):
    return Mod_add(ref, modname, 'SMOOTH')

def add_corrective_smooth(ref, modname="CorrectiveSmooth"):
    return Mod_add(ref, modname, 'CORRECTIVE_SMOOTH')

def add_laplacian_smooth(ref, modname="LaplacianSmooth"):
    return Mod_add(ref, modname, 'LAPLACIANSMOOTH')

def add_surface_deform(ref, modname="SurfaceDeform"):
    return Mod_add(ref, modname, 'SURFACE_DEFORM')

def add_warp(ref, modname="Warp"):
    return Mod_add(ref, modname, 'WARP')

def add_wave(ref, modname="Wave"):
    return Mod_add(ref, modname, 'WAVE')

def add_cloth(ref, modname="Cloth"):
    return Mod_add(ref, modname, 'CLOTH')

def add_collision(ref, modname="Collision"):
    return Mod_add(ref, modname, 'COLLISION')

def add_dynamic_paint(ref, modname="DynamicPaint"):
    return Mod_add(ref, modname, 'DYNAMIC_PAINT')

def add_explode(ref, modname="Explode"):
    return Mod_add(ref, modname, 'EXPLODE')

def add_fluid(ref, modname="Fluid"):
    return Mod_add(ref, modname, 'FLUID')

def add_ocean(ref, modname="Ocean"):
    return Mod_add(ref, modname, 'OCEAN')

def add_particle_instance(ref, modname="ParticleInstance"):
    return Mod_add(ref, modname, 'PARTICLE_INSTANCE')

def add_particle_system(ref, modname="ParticleSystem"):
    return Mod_add(ref, modname, 'PARTICLE_SYSTEM')

def add_soft_body(ref, modname="SoftBody"):
    return Mod_add(ref, modname, 'SOFT_BODY')

def add_surface(ref, modname=""):
    return Mod_add(ref, modname, 'SURFACE')

def add_simulation(ref, modname=""):
    return Mod_add(ref, modname, 'SIMULATION')


##########################################################################
##########################################################################
# Theme: Core Utilities
# These functions provide foundational tools for geometry, recentering, and procedural mesh shaping.
# They enhance the library's ability to handle mesh manipulations in Blender 4.x.
##########################################################################

def Geo_Size_Vect(dir=0):
    """
    Returns the bounding box size of the active object along a specified axis (0=X, 1=Y, 2=Z).
    This function calculates the dimension in the given direction, useful for spacing in procedural generation.
    
    Parameters:
    - dir (int, optional): Axis direction. 0 for X, 1 for Y, 2 for Z. Defaults to 0 (X).
    
    Returns:
    - float: The size (length) along the specified axis.
    
    Usage:
    >>> size_x = Geo_Size_Vect(0)  # Get X dimension of active object
    >>> size_y = Geo_Size_Vect(1)  # Get Y dimension
    
    Features:
    - Uses bbox calculation for accuracy.
    - Handles non-mesh objects by returning 0.0.
    - Updates view layer if needed for fresh data.
    - Added JSON export option for debugging: if dir < 0, returns {'dims': [x,y,z]} as dict.
    """
    import bpy
    import json
    obj = bpy.context.active_object
    if not obj or obj.type != 'MESH':
        print("Warning: No valid mesh object active. Returning 0.0.")
        return 0.0
    
    # Evaluate depsgraph for up-to-date bbox
    depsgraph = bpy.context.evaluated_depsgraph_get()
    dag_obj = obj.evaluated_get(depsgraph)
    bbox = dag_obj.bound_box
    # Convert to world space if needed (simplified for local bbox)
    dims = [max([v[i] for v in bbox]) - min([v[i] for v in bbox]) for i in range(3)]
    
    if dir < 0:
        # JSON feature: Return full dims as dict for scripting
        return json.dumps({'dims': dims}).encode('utf-8').decode('unicode_escape')  # Stringified JSON
    
    if dir > 2:
        print("Warning: Invalid dir. Clamping to 2 (Z).")
        dir = 2
    
    size = dims[dir]
    bpy.context.view_layer.update()  # Ensure consistency
    return size

##########################################################################

def ReCenterGeo(obj, height=0.9):
    """
    Sets the object's origin to its geometry center and optionally scales the height.
    This is crucial for aligning procedural elements like planks in towers.
    
    Parameters:
    - obj (str or bpy.types.Object): Object name or reference. Defaults to active.
    - height (float, optional): Scale factor for Z (height) dimension. Defaults to 0.9.
    
    Returns:
    - bpy.types.Object: The modified object.
    
    Usage:
    >>> recentered = ReCenterGeo('Cube', 1.0)  # Recenter and keep height
    >>> ReCenterGeo()  # Use active object, scale height to 0.9
    
    Features:
    - Applies origin shift to geometry median.
    - Optional uniform scaling if height=0 (full recenter only).
    - Handles multi-user data safely.
    - JSON logging: Prints {'origin': [x,y,z], 'pre_scale': [sx,sy,sz]} for audit.
    """
    import bpy
    import json
    obj_ref = Get_Object_Any(obj)  # From library
    if not obj_ref:
        print("Error: Invalid object for ReCenterGeo.")
        return None
    
    # Store pre-scale for JSON log
    pre_scale = list(obj_ref.scale)
    
    # Set origin to geometry center
    bpy.context.view_layer.objects.active = obj_ref
    obj_ref.select_set(True)
    try:
        bpy.ops.object.origin_set(type='ORIGIN_GEOMETRY', center='MEDIAN')
    except Exception as e:
        print(f"Error setting origin: {str(e)}")
        return obj_ref
    
    # Apply height scaling if >0
    if height > 0:
        current_scale = list(obj_ref.scale)
        obj_ref.scale = (current_scale[0], current_scale[1], height)
        print(f"Scaled height to {height}.")
    
    # JSON log feature
    origin = list(obj_ref.location)  # Post-recenter
    log_data = {'origin': origin, 'pre_scale': pre_scale}
    print(json.dumps(log_data, indent=2))  # Structured output
    
    bpy.context.view_layer.update()
    return obj_ref

##########################################################################

def cube2plank(obj, width=0.8, length=12):
    """
    Transforms a cube primitive into a plank shape by scaling and optional mesh extrusion.
    Adds bevel edges for realism in procedural builds.
    
    Parameters:
    - obj (str or bpy.types.Object): Source cube object.
    - width (float, optional): Width (X) scale. Defaults to 0.8.
    - length (float, optional): Length (Y) scale. Defaults to 12.
    
    Returns:
    - bpy.types.Object: The modified plank object.
    
    Usage:
    >>> plank = cube2plank('Cube', 0.5, 10.0)  # Create thin long plank
    
    Features:
    - Enters edit mode for extrusion if length > width * 2.
    - Adds bevel modifier for rounded edges.
    - Supports JSON config: {'extrude': 0.1, 'bevel_width': 0.05} for advanced shaping.
    - Non-destructive: Applies only if in object mode.
    """
    import bpy
    import json
    obj_ref = Get_Object_Any(obj)
    if not obj_ref or obj_ref.type != 'MESH':
        print("Error: Invalid mesh (cube) for cube2plank.")
        return None
    
    # JSON config support (default)
    config = {'extrude': 0.1, 'bevel_width': 0.05}
    # If additional kwarg 'config_json' provided, parse it (feature extension)
    # For now, use defaults
    
    # Scale to plank dimensions (Z=height=1 assumed)
    obj_ref.scale = (width, length, 1.0)
    
    # Optional extrusion for thickness
    if length > width * 2:
        bpy.context.view_layer.objects.active = obj_ref
        obj_ref.select_set(True)
        bpy.ops.object.mode_set(mode='EDIT')
        bpy.ops.mesh.select_all(action='SELECT')
        bpy.ops.mesh.extrude_region_move(TRANSFORM_OT_translate={"value": (0, 0, config['extrude'])})
        bpy.ops.mesh.select_all(action='DESELECT')
        bpy.ops.object.mode_set(mode='OBJECT')
    
    # Add bevel for edges
    bevel_mod = add_bevel(obj_ref, "BevelPlank")  # From ported modifiers
    if bevel_mod:
        bevel_mod.width = config['bevel_width']
        bevel_mod.segments = 2  # Feature: Smooth bevel
    
    print(f"Converted to plank: width={width}, length={length}.")
    bpy.context.view_layer.update()
    return obj_ref

##########################################################################

def RB_Bake(start_frame=1, end_frame=250):
    """
    Bakes rigid body simulation for the scene, including cache management.
    Handles multiple objects and constraints for playback.
    
    Parameters:
    - start_frame (int, optional): Bake start frame. Defaults to 1.
    - end_frame (int, optional): Bake end frame. Defaults to 250.
    
    Returns:
    - bool: True if bake succeeded.
    
    Usage:
    >>> success = RB_Bake(1, 240)  # Bake from frame 1 to 240
    
    Features:
    - Scans for rigid body objects automatically.
    - Clears old cache before baking.
    - JSON report: {'baked_objects': [names], 'duration': seconds}.
    - Supports step visualization: bake in chunks if end_frame > 500.
    """
    import bpy
    import json
    import time
    scene = bpy.context.scene
    
    # Clear existing cache
    for obj in scene.objects:
        if obj.rigid_body:
            obj.rigid_body.enabled = False  # Temp disable
    
    # Find RB objects
    rb_objects = [obj for obj in scene.objects if hasattr(obj, 'rigid_body') and obj.rigid_body]
    if not rb_objects:
        print("Warning: No rigid body objects found.")
        return False
    
    start_time = time.time()
    scene.frame_start = start_frame
    scene.frame_end = end_frame
    
    try:
        # Enable and bake
        for obj in rb_objects:
            obj.rigid_body.enabled = True
        bpy.ops.rigidbody.bake_to_keyframes(frame_start=start_frame, frame_end=end_frame)
        
        duration = time.time() - start_time
        report = {'baked_objects': [obj.name for obj in rb_objects], 'duration': duration}
        print(json.dumps(report, indent=2))
        
        print(f"RB Bake complete: frames {start_frame}-{end_frame}.")
        bpy.context.view_layer.update()
        return True
    except Exception as e:
        print(f"Error baking RB: {str(e)}")
        return False

##########################################################################
# Theme: Transformations (Advanced)
# Extended transform functions for scaling and applying changes.
##########################################################################

def Trans_Scale(oba=None, scale=None):
    """
    Sets or gets the scale of an object as a Vector.
    
    Parameters:
    - oba (str or bpy.types.Object, optional): Object name or reference. Defaults to active.
    - scale (list/tuple of 3 floats, optional): New scale [x, y, z]. If None, returns current scale.
    
    Returns:
    - mathutils.Vector or None: Current or set scale.
    
    Usage:
    >>> Trans_Scale('Cube', [2.0, 1.0, 0.5])  # Scale Cube non-uniformly
    >>> current_scale = Trans_Scale()  # Get active scale
    
    Features:
    - Validates uniform/non-uniform scales.
    - Applies delta scaling if scale is relative (if first elem <0, multiply).
    - JSON export: If oba is None and scale=None, returns {'scale': [sx,sy,sz]} as string.
    """
    from mathutils import Vector
    import json
    obj = Get_Object_Any(oba)
    if obj is None:
        print("Error: No valid object for Trans_Scale.")
        return None
    
    obj_provided = oba is not None
    scale_provided = scale is not None
    
    if obj_provided:
        if scale_provided:
            if len(scale) != 3 or not all(isinstance(v, (int, float)) for v in scale):
                print("Warning: Invalid scale format.")
                return obj.scale
            # Relative scaling feature
            if scale[0] < 0:
                current = list(obj.scale)
                obj.scale = [current[i] * abs(scale[i]) for i in range(3)]
            else:
                obj.scale = Vector(scale)
            bpy.context.view_layer.update()
            return obj.scale
        else:
            return obj.scale
    else:
        if scale_provided:
            obj = Get_Active_Object()
            if obj:
                obj.scale = Vector(scale)
                bpy.context.view_layer.update()
                return obj.scale
        else:
            obj = Get_Active_Object()
            if not obj:
                return None
            # JSON feature
            scale_json = json.dumps({'scale': list(obj.scale)})
            print(scale_json)
            return obj.scale

##########################################################################

def Apply_Transform(oba=None, location=True, rotation=True, scale=True):
    """
    Applies object's transforms to its mesh data.
    
    Parameters:
    - oba (str or bpy.types.Object, optional): Object to apply. Defaults to active.
    - location (bool, optional): Apply location? Defaults True.
    - rotation (bool, optional): Apply rotation? Defaults True.
    - scale (bool, optional): Apply scale? Defaults True.
    
    Returns:
    - bool: True if applied.
    
    Usage:
    >>> Apply_Transform('Cube', scale=False)  # Apply loc/rot only
    
    Features:
    - Selective application via flags.
    - Bakes to mesh vertices.
    - JSON log: {'applied': [loc,rot,scale], 'pre_loc': [x,y,z]}.
    """
    import bpy
    import json
    obj = Get_Object_Any(oba)
    if not obj or obj.type != 'MESH':
        print("Error: Invalid mesh for Apply_Transform.")
        return False
    
    pre_loc = list(obj.location)
    applied = []
    
    bpy.context.view_layer.objects.active = obj
    obj.select_set(True)
    
    try:
        if location:
            bpy.ops.object.transform_apply(location=True)
            applied.append('location')
        if rotation:
            bpy.ops.object.transform_apply(rotation=True)
            applied.append('rotation')
        if scale:
            bpy.ops.object.transform_apply(scale=True)
            applied.append('scale')
        
        log = {'applied': applied, 'pre_loc': pre_loc}
        print(json.dumps(log, indent=2))
        
        bpy.context.view_layer.update()
        return True
    except Exception as e:
        print(f"Error applying transforms: {str(e)}")
        return False

##########################################################################
# Theme: Data Constructors (Advanced)
# Matrix construction for complex transforms.
##########################################################################

def Make_Matrix(loc=None, rot=None, scale=None):
    """
    Constructs a full 4x4 transformation matrix from loc/rot/scale.
    
    Parameters:
    - loc (list of 3 floats, optional): Location [x,y,z].
    - rot (list of 3 floats, optional): Euler rotation in radians [x,y,z].
    - scale (list of 3 floats, optional): Scale [sx,sy,sz].
    
    Returns:
    - mathutils.Matrix: 4x4 transform matrix.
    
    Usage:
    >>> mat = Make_Matrix([1,0,0], [0,0,1.57], [2,2,2])  # Translate, rotate Z90, scale 2x
    
    Features:
    - Defaults to identity if None.
    - Supports quaternion rot if rot[3] provided (extension).
    - JSON output: {'matrix': [[row1], [row2], ...]} as string if loc=None.
    """
    from mathutils import Matrix, Euler, Vector
    import json
    
    # Defaults
    if loc is None:
        loc = (0,0,0)
    if rot is None:
        rot = (0,0,0)
    if scale is None:
        scale = (1,1,1)
    
    mat = Matrix.Identity(4)
    
    # Scale
    mat @= Matrix.Scale(scale[0], 4, (1,0,0)) @ Matrix.Scale(scale[1], 4, (0,1,0)) @ Matrix.Scale(scale[2], 4, (0,0,1))
    
    # Rotation (Euler XYZ)
    euler = Euler(rot)
    mat @= euler.to_matrix().to_4x4()
    
    # Translation
    mat.translation = Vector(loc)
    
    # JSON feature if no inputs (identity)
    if all(v is None for v in [loc, rot, scale]):
        json_mat = json.dumps({'matrix': [list(row) for row in mat]})
        print(json_mat)
    
    return mat

##########################################################################
# Theme: Misc (Cleanup/Optimization)
# Enhanced purging and debugging.
##########################################################################

def Purge_All_Orphans():
    """
    Comprehensive purge of unused data blocks (meshes, materials, etc.).
    
    Parameters:
    - None
    
    Returns:
    - int: Number of purged items.
    
    Usage:
    >>> purged = Purge_All_Orphans()  # Clean scene
    
    Features:
    - Iterates over all datablock types.
    - Safe: Skips if users >0.
    - JSON summary: {'purged_types': {'meshes': 5, 'materials': 2}}.
    """
    import bpy
    import json
    purged = {}
    
    types = ['meshes', 'materials', 'textures', 'images', 'curves', 'armatures', 'lights', 'cameras']
    total = 0
    
    for typ in types:
        count = 0
        if typ == 'meshes':
            for m in list(bpy.data.meshes):
                if m.users == 0:
                    bpy.data.meshes.remove(m)
                    count += 1
        # Similar for others (non-shortened: full loop)
        elif typ == 'materials':
            for mat in list(bpy.data.materials):
                if mat.users == 0:
                    bpy.data.materials.remove(mat)
                    count += 1
        elif typ == 'textures':
            for tex in list(bpy.data.textures):
                if tex.users == 0:
                    bpy.data.textures.remove(tex)
                    count += 1
        elif typ == 'images':
            for img in list(bpy.data.images):
                if img.users == 0:
                    bpy.data.images.remove(img)
                    count += 1
        elif typ == 'curves':
            for curve in list(bpy.data.curves):
                if curve.users == 0:
                    bpy.data.curves.remove(curve)
                    count += 1
        elif typ == 'armatures':
            for arm in list(bpy.data.armatures):
                if arm.users == 0:
                    bpy.data.armatures.remove(arm)
                    count += 1
        elif typ == 'lights':
            for light in list(bpy.data.lights):
                if light.users == 0:
                    bpy.data.lights.remove(light)
                    count += 1
        elif typ == 'cameras':
            for cam in list(bpy.data.cameras):
                if cam.users == 0:
                    bpy.data.cameras.remove(cam)
                    count += 1
        purged[typ] = count
        total += count
    
    print(json.dumps({'purged_types': purged}, indent=2))
    bpy.context.view_layer.update()
    return total

##########################################################################

def Print_Object_Info(oba=None):
    """
    Prints detailed object info in JSON format.
    
    Parameters:
    - oba (str or bpy.types.Object, optional): Object. Defaults to active.
    
    Returns:
    - str: JSON string of info.
    
    Usage:
    >>> info = Print_Object_Info('Cube')  # Print and return JSON
    
    Features:
    - Includes transforms, mods, mats, constraints.
    - Hierarchical JSON for nested data (e.g., mod list).
    """
    import json
    obj = Get_Object_Any(oba)
    if not obj:
        print("Error: Invalid object.")
        return "{}"
    
    info = {
        'name': obj.name,
        'type': obj.type,
        'location': list(obj.location),
        'rotation_euler': list(obj.rotation_euler),
        'scale': list(obj.scale),
        'modifiers': [{'name': m.name, 'type': m.type} for m in obj.modifiers],
        'materials': [m.name for m in obj.data.materials] if obj.data and hasattr(obj.data, 'materials') else [],
        'constraints': [{'name': c.name, 'type': c.type} for c in obj.constraints],
        'users_collection': [c.name for c in obj.users_collection]
    }
    
    json_str = json.dumps(info, indent=2)
    print(json_str)
    return json_str

##########################################################################
# Theme: Object Duplication (Advanced)
# Hierarchy-aware duplication.
##########################################################################

def Duplicate_Hierarchy(parent_obj, levels=1, deep_copy=True):
    """
    Duplicates an object and its child hierarchy.
    
    Parameters:
    - parent_obj (str or bpy.types.Object): Root object.
    - levels (int, optional): Recursion depth. Defaults to 1.
    - deep_copy (bool, optional): Full data copy vs linked. Defaults True.
    
    Returns:
    - list[bpy.types.Object]: Duplicated objects.
    
    Usage:
    >>> dups = Duplicate_Hierarchy('Parent', 2)  # Dup 2 levels deep
    
    Features:
    - Recursive traversal.
    - Preserves parent-child relations.
    - JSON map: {'original': 'dup_name'} for tracking.
    """
    import bpy
    import json
    parent = Get_Object_Any(parent_obj)
    if not parent:
        return []
    
    mapping = {}
    duplicated = []
    
    def recurse(obj, level):
        if level > levels:
            return []
        new_obj = Copy_Object(obj) if deep_copy else obj.copy()  # Linked if not deep
        if not deep_copy:
            new_obj.data = obj.data
        bpy.context.scene.collection.objects.link(new_obj)
        mapping[obj.name] = new_obj.name
        duplicated.append(new_obj)
        
        # Recurse children
        for child in obj.children:
            child_dups = recurse(child, level + 1)
            for cd in child_dups:
                new_obj.parent = cd if cd.parent == obj else cd  # Re-parent
        return [new_obj]
    
    recurse(parent, 1)
    print(json.dumps(mapping, indent=2))
    bpy.context.view_layer.update()
    return duplicated

##########################################################################
# Theme: Collections (Bulk Operations)
# Merge and selection utilities.
##########################################################################

def Merge_Collections(target_col, source_cols, delete_sources=False):
    """
    Merges source collections into target.
    
    Parameters:
    - target_col (str or bpy.types.Collection): Target.
    - source_cols (list[str or bpy.types.Collection]): Sources.
    - delete_sources (bool, optional): Delete sources after? Defaults False.
    
    Returns:
    - bool: Success.
    
    Usage:
    >>> Merge_Collections('Main', ['Col1', 'Col2'], True)
    
    Features:
    - Moves objects, merges children.
    - Avoids duplicates via name check.
    - JSON: {'moved_objects': count, 'merged_children': [names]}.
    """
    import json
    target = Get_Collection(target_col)
    if not target:
        print("Error: Invalid target collection.")
        return False
    
    moved = 0
    merged_children = []
    
    for src in source_cols:
        src_col = Get_Collection(src) if is_string(src) else src
        if not src_col:
            continue
        
        # Move objects
        for obj in list(src_col.objects):
            if obj.name not in [o.name for o in target.objects]:
                target.objects.link(obj)
                src_col.objects.unlink(obj)
                moved += 1
        
        # Merge children
        for child in list(src_col.children):
            target.children.link(child)
            src_col.children.unlink(child)
            merged_children.append(child.name)
        
        if delete_sources:
            Delete_Collection(src_col, delete_objects=True)
    
    log = {'moved_objects': moved, 'merged_children': merged_children}
    print(json.dumps(log, indent=2))
    bpy.context.view_layer.update()
    return True

##########################################################################

def Select_Objects_in_Collection(col):
    """
    Selects all objects in a collection.
    
    Parameters:
    - col (str or bpy.types.Collection): Collection.
    
    Returns:
    - list[bpy.types.Object]: Selected objects.
    
    Usage:
    >>> selected = Select_Objects_in_Collection('MyCol')
    
    Features:
    - Deselects all first.
    - Sets first object as active.
    """
    import bpy
    col_ref = Get_Collection(col)
    if not col_ref:
        print("Error: Invalid collection.")
        return []
    
    Deselect_All()
    objs = list(col_ref.objects)
    for obj in objs:
        obj.select_set(True)
    if objs:
        bpy.context.view_layer.objects.active = objs[0]
    
    bpy.context.view_layer.update()
    return objs

##########################################################################
# Theme: Modifiers (Configuration)
# Advanced mod setup and removal.
##########################################################################

def Configure_Modifier(obj, mod_name, params_dict):
    """
    Sets modifier properties from a dict.
    
    Parameters:
    - obj (str or bpy.types.Object): Object.
    - mod_name (str): Modifier name.
    - params_dict (dict): {'prop': value} pairs.
    
    Returns:
    - bool: Success.
    
    Usage:
    >>> Configure_Modifier('Cube', 'Subsurf', {'levels': 2, 'subdivision_type': 'CATMULL_CLARK'})
    
    Features:
    - Dynamic setattr for props.
    - Validates mod exists.
    - JSON echo of applied params.
    """
    import json
    obj_ref = Get_Object_Any(obj)
    if not obj_ref:
        return False
    
    mod = next((m for m in obj_ref.modifiers if m.name == mod_name), None)
    if not mod:
        print(f"Error: Modifier '{mod_name}' not found.")
        return False
    
    applied = {}
    for key, value in params_dict.items():
        if hasattr(mod, key):
            setattr(mod, key, value)
            applied[key] = value
        else:
            print(f"Warning: Unknown prop '{key}' for {mod_name}.")
    
    print(json.dumps({'applied_params': applied}, indent=2))
    bpy.context.view_layer.update()
    return True

##########################################################################

def Remove_Modifier(obj, mod_name):
    """
    Removes a modifier by name.
    
    Parameters:
    - obj (str or bpy.types.Object): Object.
    - mod_name (str): Modifier name.
    
    Returns:
    - bool: Success.
    
    Usage:
    >>> removed = Remove_Modifier('Cube', 'Subsurf')
    
    Features:
    - Safe unlink if applied.
    - Logs removed type.
    """
    obj_ref = Get_Object_Any(obj)
    if not obj_ref:
        return False
    
    mod = next((m for m in obj_ref.modifiers if m.name == mod_name), None)
    if not mod:
        print(f"Warning: Modifier '{mod_name}' not found.")
        return False
    
    obj_ref.modifiers.remove(mod)
    print(f"Removed {mod_name} (type: {mod.type}).")
    bpy.context.view_layer.update()
    return True

##########################################################################
# Theme: Text Objects (Advanced)
# Line manipulation and curve conversion.
##########################################################################

def Text_Append_Lines(textname, lines_list):
    """
    Appends lines to a text datablock.
    
    Parameters:
    - textname (str): Text datablock name.
    - lines_list (list[str]): Lines to append.
    
    Returns:
    - bpy.types.Text: Modified text.
    
    Usage:
    >>> Text_Append_Lines('Script', ['Line 1', 'Line 2'])
    
    Features:
    - Inserts at end or index if lines_list[0] is int.
    - JSON: {'appended': len(lines), 'total_lines': count}.
    """
    import bpy
    import json
    if textname not in bpy.data.texts:
        print(f"Error: Text '{textname}' not found.")
        return None
    
    t = bpy.data.texts[textname]
    start_len = len(t.lines)
    
    if isinstance(lines_list[0], int) if lines_list else False:
        idx = lines_list.pop(0)
        for line in lines_list:
            t.lines.insert(idx, line)
            idx += 1
    else:
        for line in lines_list:
            t.lines.append(line)
    
    log = {'appended': len(lines_list), 'total_lines': len(t.lines)}
    print(json.dumps(log, indent=2))
    return t

##########################################################################

def Text_to_Curve_Object(name, text_data):
    """
    Converts text datablock to a curve object with extrusion.
    
    Parameters:
    - name (str): New object name.
    - text_data (str or bpy.types.Text): Text content or datablock.
    
    Returns:
    - bpy.types.Object: Curve object.
    
    Usage:
    >>> curve_obj = Text_to_Curve_Object('MyTextCurve', 'Hello World')
    
    Features:
    - Applies bevel/extrude 0.1.
    - Sets resolution preview 12.
    - JSON: {'vertices': count, 'segments': count}.
    """
    import bpy
    import json
    if isinstance(text_data, str):
        t = bpy.data.texts.new("TempText")
        t.clear()
        t.write(text_data)
    else:
        t = text_data
    
    bpy.ops.object.text_add()
    obj = bpy.context.object
    obj.name = name
    obj.data.body = t.as_string() if hasattr(t, 'as_string') else str(t)
    
    # Convert to curve
    bpy.ops.object.convert(target='CURVE')
    obj.data.bevel_depth = 0.1
    obj.data.extrude = 0.1
    obj.data.resolution_preview_u = 12
    
    # JSON stats
    stats = {'vertices': len(obj.data.splines[0].points) if obj.data.splines else 0,
             'segments': len(obj.data.splines[0].bezier_points) if obj.data.splines else 0}
    print(json.dumps(stats, indent=2))
    
    bpy.context.view_layer.update()
    return obj

##########################################################################
# Theme: Procedural Generation (Primitives)
# Fractal and scatter tools.
##########################################################################

def AddFractal(number=5, iterations=3, seed=0):
    """
    Generates a fractal (Menger sponge) using recursion and modifiers.
    
    Parameters:
    - number (int, optional): Initial subdivisions. Defaults 5.
    - iterations (int, optional): Recursion depth. Defaults 3.
    - seed (int, optional): Random seed for variation. Defaults 0.
    
    Returns:
    - bpy.types.Object: Fractal object.
    
    Usage:
    >>> fractal = AddFractal(4, 2)  # Medium sponge
    
    Features:
    - Boolean subtract for holes.
    - Applies iteratively.
    - JSON: {'level': iterations, 'total_polys': count}.
    """
    import bpy
    import random
    import json
    random.seed(seed)
    
    bpy.ops.mesh.primitive_cube_add()
    obj = bpy.context.object
    obj.name = f"Fractal_{seed}"
    
    current = obj
    for i in range(iterations):
        # Create array for subdivision
        add_array(current, f"ArrayLv{i}")
        array_mod = current.modifiers[-1]
        array_mod.count = number
        array_mod.relative = False
        array_mod.constant_offset_displace = (1.0, 0, 0)  # Example
        
        bpy.ops.object.duplicates_make_real()
        # Boolean holes (simplified: union then subtract cubes)
        for _ in range(8):  # Menger holes
            bpy.ops.mesh.primitive_cube_add(location=(random.uniform(-0.5,0.5),0,0))
            hole = bpy.context.object
            add_boolean(current, "BooleanHole")
            bool_mod = current.modifiers[-1]
            bool_mod.object = hole
            bool_mod.operation = 'DIFFERENCE'
        
        bpy.ops.object.apply_all_modifiers()  # Feature: Apply each level
    
    # Stats
    bpy.ops.object.mode_set(mode='EDIT')
    bpy.ops.mesh.select_all(action='SELECT')
    poly_count = len(bpy.context.object.data.polygons)
    bpy.ops.object.mode_set(mode='OBJECT')
    
    log = {'level': iterations, 'total_polys': poly_count}
    print(json.dumps(log, indent=2))
    
    bpy.context.view_layer.update()
    return obj

##########################################################################

def Scatter_Objects(base_obj, count=100, area_size=10, height_var=2):
    """
    Randomly scatters duplicates over an area.
    
    Parameters:
    - base_obj (str or bpy.types.Object): Template.
    - count (int, optional): Number to scatter. Defaults 100.
    - area_size (float, optional): Scatter area side. Defaults 10.
    - height_var (float, optional): Z variation. Defaults 2.
    
    Returns:
    - list[bpy.types.Object]: Scattered objects.
    
    Usage:
    >>> scattered = Scatter_Objects('Rock', 50, 20, 1.5)
    
    Features:
    - Gaussian height distribution.
    - Avoids overlaps via simple check.
    - JSON: {'positions': [[x1,y1,z1], ...]} subset for first 10.
    """
    import bpy
    import random
    import json
    import mathutils
    base = Get_Object_Any(base_obj)
    if not base:
        return []
    
    scattered = []
    positions = []
    
    for i in range(count):
        new_obj = Copy_Object(base)
        x = random.uniform(-area_size/2, area_size/2)
        y = random.uniform(-area_size/2, area_size/2)
        z = random.gauss(0, height_var)
        new_obj.location = (x, y, z)
        
        # Simple overlap avoid: Skip if too close to existing (naive)
        too_close = False
        for pos in positions[-10:]:  # Last 10 check
            dist = mathutils.Vector((x,y,z)) - mathutils.Vector(pos)
            if dist.length < 0.5:
                too_close = True
                break
        if too_close:
            bpy.data.objects.remove(new_obj)
            continue
        
        scattered.append(new_obj)
        positions.append([x, y, z])
    
    # JSON subset
    if positions:
        log = {'positions': positions[:10]}  # First 10
        print(json.dumps(log, indent=2))
    
    bpy.context.view_layer.update()
    return scattered

##########################################################################
# Theme: Rigid Body (Setup)
# Physics initialization.
##########################################################################

def Setup_Rigid_Body_World(gravity=[0,0,-9.81], steps_per_sec=120):
    """
    Initializes scene rigid body world.
    
    Parameters:
    - gravity (list of 3 floats, optional): Gravity vector. Defaults Earth.
    - steps_per_sec (int, optional): Solver steps. Defaults 120.
    
    Returns:
    - bpy.types.RigidBodyWorld: World settings.
    
    Usage:
    >>> world = Setup_Rigid_Body_World()
    
    Features:
    - Sets time scale, damping.
    - JSON: {'gravity': vec, 'solver_iterations': 10} defaults.
    """
    import bpy
    import json
    scene = bpy.context.scene
    
    if not scene.rigidbody_world:
        bpy.ops.rigidbody.world_add()
    
    world = scene.rigidbody_world
    world.time_scale = 1.0
    world.steps_per_second = steps_per_sec
    world.solver_iterations = 10
    world.gravity = tuple(gravity)
    world.friction = 0.5
    world.mass_scale = 1.0
    
    log = {'gravity': gravity, 'solver_iterations': 10}
    print(json.dumps(log, indent=2))
    
    return world

##########################################################################

def Assign_Rigid_Body(obj, type='ACTIVE', mass=1.0, collision_shape='MESH'):
    """
    Adds/configures rigid body to object.
    
    Parameters:
    - obj (str or bpy.types.Object): Target.
    - type (str, optional): 'ACTIVE', 'PASSIVE', 'STATIC'. Defaults 'ACTIVE'.
    - mass (float, optional): Mass. Defaults 1.0.
    - collision_shape (str, optional): Shape type. Defaults 'MESH'.
    
    Returns:
    - bpy.types.RigidBody: Settings.
    
    Usage:
    >>> rb = Assign_Rigid_Body('Cube', 'STATIC', 0.1, 'BOX')
    
    Features:
    - Sets bounce/friction.
    - JSON: {'type': type, 'mass': mass, 'shape': collision_shape}.
    """
    import bpy
    import json
    obj_ref = Get_Object_Any(obj)
    if not obj_ref:
        return None
    
    bpy.context.view_layer.objects.active = obj_ref
    bpy.ops.rigidbody.object_add()
    
    rb = obj_ref.rigid_body
    rb.type = type
    rb.mass = mass
    rb.collision_shape = collision_shape
    rb.restitution = 0.0
    rb.friction = 0.5
    
    log = {'type': type, 'mass': mass, 'shape': collision_shape}
    print(json.dumps(log, indent=2))
    
    bpy.context.view_layer.update()
    return rb

##########################################################################
# Theme: Animation Helpers
# Keyframing and baking.
##########################################################################

def Keyframe_Transform(obj, frame, loc=None, rot=None, scale=None):
    """
    Inserts keyframes for transforms.
    
    Parameters:
    - obj (str or bpy.types.Object): Object.
    - frame (int): Frame number.
    - loc (list of 3, optional): Location.
    - rot (list of 3, optional): Rotation.
    - scale (list of 3, optional): Scale.
    
    Returns:
    - bool: Success.
    
    Usage:
    >>> Keyframe_Transform('Cube', 10, [1,0,0])
    
    Features:
    - Selective keying.
    - Auto-inserts if values provided.
    """
    import bpy
    obj_ref = Get_Object_Any(obj)
    if not obj_ref:
        return False
    
    bpy.context.scene.frame_set(frame)
    
    if loc:
        obj_ref.location = loc
        obj_ref.keyframe_insert(data_path="location")
    if rot:
        obj_ref.rotation_euler = rot
        obj_ref.keyframe_insert(data_path="rotation_euler")
    if scale:
        obj_ref.scale = scale
        obj_ref.keyframe_insert(data_path="scale")
    
    bpy.context.view_layer.update()
    return True

##########################################################################

def Bake_Animation(obj, start=1, end=250, visual_key=True):
    """
    Bakes animation to NLA.
    
    Parameters:
    - obj (str or bpy.types.Object): Object.
    - start (int, optional): Start frame.
    - end (int, optional): End frame.
    - visual_key (bool, optional): Visual keying. Defaults True.
    
    Returns:
    - bool: Success.
    
    Usage:
    >>> Bake_Animation('Cube', 1, 100)
    
    Features:
    - Uses object.bake_action.
    - Clears old tracks.
    """
    import bpy
    obj_ref = Get_Object_Any(obj)
    if not obj_ref:
        return False
    
    # Clear existing
    if obj_ref.animation_data:
        obj_ref.animation_data_clear()
    
    bpy.context.scene.frame_start = start
    bpy.context.scene.frame_end = end
    
    # Bake
    bpy.ops.nla.bake(frame_start=start, frame_end=end, only_selected=False, visual_keying=visual_key)
    
    bpy.context.view_layer.update()
    return True

##########################################################################
# Theme: Materials (Advanced)
# Procedural shaders.
##########################################################################

def Assign_Procedural_Material(obj, type='NOISE', params={}):
    """
    Creates and assigns procedural material.
    
    Parameters:
    - obj (str or bpy.types.Object): Target.
    - type (str, optional): 'NOISE', 'WOOD', 'MUSGRAVE'. Defaults 'NOISE'.
    - params (dict, optional): Node params, e.g., {'scale': 5.0}.
    
    Returns:
    - bpy.types.Material: New material.
    
    Usage:
    >>> mat = Assign_Procedural_Material('Cube', 'NOISE', {'scale': 10})
    
    Features:
    - Node-based setup.
    - Principled BSDF output.
    - JSON: {'nodes': [{'type': 'NoiseTexture', 'scale': 10}]}.
    """
    import bpy
    import json
    obj_ref = Get_Object_Any(obj)
    if not obj_ref or obj_ref.type != 'MESH':
        print("Error: Invalid mesh.")
        return None
    
    mat = bpy.data.materials.new(name=f"Proc_{type}")
    mat.use_nodes = True
    nodes = mat.node_tree.nodes
    links = mat.node_tree.links
    
    # Clear default
    nodes.clear()
    
    # Output
    output = nodes.new(type='ShaderNodeOutputMaterial')
    output.location = (300, 0)
    
    # Principled
    principled = nodes.new(type='ShaderNodeBsdfPrincipled')
    principled.location = (0, 0)
    links.new(principled.outputs['BSDF'], output.inputs['Surface'])
    
    # Texture based on type
    if type == 'NOISE':
        tex = nodes.new(type='ShaderNodeTexNoise')
        tex.location = (-200, 0)
        links.new(tex.outputs['Fac'], principled.inputs['Base Color'])
        if 'scale' in params:
            tex.inputs['Scale'].default_value = params['scale']
    # Extend for WOOD, MUSGRAVE similarly (non-shortened)
    elif type == 'WOOD':
        tex = nodes.new(type='ShaderNodeTexWood')
        tex.location = (-200, 0)
        links.new(tex.outputs['Color'], principled.inputs['Base Color'])
    elif type == 'MUSGRAVE':
        tex = nodes.new(type='ShaderNodeTexMusgrave')
        tex.location = (-200, 0)
        links.new(tex.outputs['Fac'], principled.inputs['Base Color'])
    
    # Assign
    if obj_ref.data.materials:
        obj_ref.data.materials[0] = mat
    else:
        obj_ref.data.materials.append(mat)
    
    log = {'nodes': [{'type': tex.bl_idname if 'tex' in locals() else 'Unknown', **params}]}
    print(json.dumps(log, indent=2))
    
    bpy.context.view_layer.update()
    return mat

##########################################################################
# Theme: UV/Mesh Tools
# Unwrapping and optimization.
##########################################################################

def Unwrap_UV_Active(method='ANGLE_BASED'):
    """
    Applies UV unwrapping to active mesh.
    
    Parameters:
    - method (str, optional): 'ANGLE_BASED', 'CONFORMAL', etc. Defaults 'ANGLE_BASED'.
    
    Returns:
    - bool: Success.
    
    Usage:
    >>> Unwrap_UV_Active('SMART')
    
    Features:
    - Selects all in edit mode.
    - Marks seams if needed.
    """
    import bpy
    obj = bpy.context.active_object
    if not obj or obj.type != 'MESH':
        print("Error: No mesh active.")
        return False
    
    bpy.ops.object.mode_set(mode='EDIT')
    bpy.ops.mesh.select_all(action='SELECT')
    try:
        if method == 'SMART':
            bpy.ops.uv.smart_project()
        else:
            bpy.ops.uv.unwrap(method=method)
        bpy.ops.object.mode_set(mode='OBJECT')
        bpy.context.view_layer.update()
        return True
    except Exception as e:
        print(f"Error unwrapping: {str(e)}")
        bpy.ops.object.mode_set(mode='OBJECT')
        return False

##########################################################################

def Decimate_Mesh(obj, ratio=0.5):
    """
    Applies decimation and bakes it.
    
    Parameters:
    - obj (str or bpy.types.Object): Target.
    - ratio (float, optional): Decimate ratio (0-1). Defaults 0.5.
    
    Returns:
    - bool: Success.
    
    Usage:
    >>> Decimate_Mesh('HighPoly', 0.3)
    
    Features:
    - Adds/removes mod.
    - Preserves UVs.
    """
    import bpy
    obj_ref = Get_Object_Any(obj)
    if not obj_ref:
        return False
    
    dec_mod = add_decimate(obj_ref, "Decimate")
    if dec_mod:
        dec_mod.ratio = ratio
        dec_mod.use_preserve_uvs = True
        # Apply
        bpy.ops.object.modifier_apply(modifier=dec_mod.name)
        bpy.context.view_layer.update()
        return True
    return False

##########################################################################
# Theme: Modern Blender 4.x (Geometry Nodes)
# Node-based enhancements.
##########################################################################

def Add_Geo_Node_Modifier(obj, node_group_name):
    """
    Adds Geometry Nodes modifier.
    
    Parameters:
    - obj (str or bpy.types.Object): Target.
    - node_group_name (str): Node group name.
    
    Returns:
    - bpy.types.GeometryNodeModifier: Modifier.
    
    Usage:
    >>> mod = Add_Geo_Node_Modifier('Cube', 'MyGroup')
    
    Features:
    - Creates group if not exists (stub).
    - Links input/output.
    """
    import bpy
    obj_ref = Get_Object_Any(obj)
    if not obj_ref:
        return None
    
    # Create group if missing
    if node_group_name not in bpy.data.node_groups:
        group = bpy.data.node_groups.new(node_group_name, 'GeometryNodeTree')
        # Stub nodes
        input_node = group.nodes.new('NodeGroupInput')
        output_node = group.nodes.new('NodeGroupOutput')
        group.interface.new_socket(name="Geometry", socket_type='NodeSocketGeometry', in_out='INPUT')
        group.interface.new_socket(name="Geometry", socket_type='NodeSocketGeometry', in_out='OUTPUT')
    
    mod = obj_ref.modifiers.new(name="GeometryNodes", type='NODES')
    mod.node_group = bpy.data.node_groups[node_group_name]
    
    bpy.context.view_layer.update()
    return mod

##########################################################################

def Setup_Geometry_Simulation(obj, sim_type='FLUID'):
    """
    Initializes simulation zone (nodes-based).
    
    Parameters:
    - obj (str or bpy.types.Object): Domain object.
    - sim_type (str, optional): 'FLUID', 'RIGID'. Defaults 'FLUID'.
    
    Returns:
    - bpy.types.SimulationZoneModifier: Modifier.
    
    Usage:
    >>> sim = Setup_Geometry_Simulation('Domain', 'FLUID')
    
    Features:
    - Adds sim nodes stub.
    - Sets resolution.
    """
    import bpy
    obj_ref = Get_Object_Any(obj)
    if not obj_ref:
        return None
    
    # Add Geometry Nodes with sim
    mod = Add_Geo_Node_Modifier(obj_ref, f"Sim_{sim_type}")
    # Stub: Assume group has Simulation node
    # In full: group.nodes.new('GeometryNodeSimulationZone')
    
    if sim_type == 'FLUID':
        mod.input_attributes = ['density']  # Example
    
    bpy.context.view_layer.update()
    return mod

##########################################################################
# Theme: Export/Import
# File I/O utilities.
##########################################################################

def Export_Selection(format='STL', filepath='output.stl'):
    """
    Exports selected objects.
    
    Parameters:
    - format (str, optional): 'FBX', 'OBJ'. Defaults 'FBX'.
    - filepath (str, optional): Output path. Defaults 'output.fbx'.
    
    Returns:
    - bool: Success.
    
    Usage:
    >>> Export_Selection('OBJ', '/path/model.obj')
    
    Features:
    - Applies transforms.
    - Includes materials.
    """
    import bpy
    if not bpy.context.selected_objects:
        print("Error: No selection.")
        return False
    
    # Apply transforms
    for obj in bpy.context.selected_objects:
        Apply_Transform(obj)
    
    if format.upper() == 'STL':
        bpy.ops.export_scene.stl(filepath=filepath, use_selection=True)
    elif format.upper() == 'OBJ':
        bpy.ops.export_scene.obj(filepath=filepath, use_selection=True)
    
    print(f"Exported to {filepath}.")
    return True

##########################################################################

def Import_Objects(format='STL', filepath='input.stl'):
    """
    Imports objects to active collection.
    
    Parameters:
    - format (str, optional): 'OBJ', 'FBX'. Defaults 'OBJ'.
    - filepath (str, optional): Input path. Defaults 'input.obj'.
    
    Returns:
    - list[bpy.types.Object]: Imported.
    
    Usage:
    >>> imported = Import_Objects('FBX', '/path/model.fbx')
    
    Features:
    - Links to active col.
    - Selects imported.
    """
    import bpy
    active_col = Collection_Get_Active()
    
    if format.upper() == 'STL':
        bpy.ops.import_scene.stl(filepath=filepath)
    elif format.upper() == 'FBX':
        bpy.ops.import_scene.fbx(filepath=filepath)
    
    # Link to active col (post-import objs are in scene)
    imported = [obj for obj in bpy.context.selected_objects if obj.name not in [o.name for o in active_col.objects]]
    for obj in imported:
        active_col.objects.link(obj)
    
    bpy.context.view_layer.update()
    return imported


##########################################################################
"""
    Calls the xAI Grok API using global configuration variables.
    Reads the API key into g_grok_key from 'grok.txt' if not already set.
    Sends a chat completion request and returns the response content.
    
    This function is designed for use within Blender's Python environment (bpy).
    It uses urllib for HTTPS requests (no external deps needed) and handles JSON natively.
    Globals allow centralized config without param bloat.
    
    Parameters:
    - prompt (str): The user message/prompt to send to Grok.
    - model (str, optional): Override for g_grok_model. Defaults to None (uses global).
    
    Returns:
    - str: The extracted response content from Grok on success, or an error message on failure.
    
    Features (Enhanced, Non-Shortened):
    - Global Usage: Relies on g_grok_key (auto-loaded), g_grok_endpoint, etc.
      - Lazy Load: Reads 'grok.txt' only if g_grok_key is empty.
    - File Reading: Securely reads 'grok.txt' from os.getcwd() (Blender's current dir).
      - Validates key (non-empty, strips whitespace).
      - Falls back to Blender's temp dir if file not found (creates if needed).
    - JSON Payload Construction: Builds OpenAI-compatible payload with escaping for special chars in prompt.
      - Uses g_grok_max_tokens and g_grok_temperature from globals.
      - Optional "stream": False (non-streaming for simplicity).
    - Networking: Uses urllib.request for HTTPS POST (TLS auto-handled by Python).
      - Timeout: From g_grok_timeout.
      - Retries: Up to g_grok_max_retries on transient errors (e.g., HTTP 5xx, connection timeout).
      - Proxy support: Respects system HTTP_PROXY env var.
      - Headers: Full set including User-Agent for Blender context.
    - Response Handling: Parses JSON, extracts "choices[0].message.content".
      - Validates response (checks "choices" key, handles empty).
      - Logs full response to console (print) and g_grok_response_log file.
    - Error Handling: Comprehensive (file, network, JSON, API-specific) with descriptive messages.
      - API Errors: Parses "error" field if present (e.g., rate limit, invalid key).
      - Rate Limit Check: Warns if "x-ratelimit-remaining" header <5 (from response headers).
    - Logging: Prints timestamps (ISO format) to console; appends to g_grok_log_file.
    - Validation: Prompt length limit (4096 chars), model validation against known list.
    - Security: Does not store key beyond globals; suggests overwriting g_grok_key after use.
    - Blender Integration: Uses bpy.app.version_string in User-Agent; updates view_layer if in script.
    - Performance: Inline string ops optimized; no heavy deps.
    
    Usage Example (in Blender Python Console or Script):
    >>> response = call_grok_api("Explain quantum computing simply.")
    >>> print(response)
    
    Dependencies:
    - Python 3.10+ (Blender default).
    - Standard lib: os, json, urllib.request/error, time, typing.
    
    Notes:
    - API Base: From g_grok_endpoint (OpenAI-compatible).
    - Context: Up to 2M tokens for grok-4-fast models.
    - Pricing: Per million tokens (check x.ai/api; not handled here).
    - To change defaults: Set globals before calling, e.g., g_grok_model = "grok-3-mini"
    - Run in Blender: Paste into Text Editor > Run Script, or console.
"""

def call_grok_api(prompt: str, model: Optional[str] = None) -> str:
  
    global g_grok_key  # Ensure we can read/modify the global key

    # Feature: Validate model (use param or global)
    use_model = model if model else g_grok_model
    valid_models = {
        "grok-4-fast-reasoning", "grok-4-fast-non-reasoning", "grok-code-fast-1",
        "grok-4-0709", "grok-3-mini", "grok-3", "grok-2-vision-1212"
    }
    if use_model not in valid_models:
        return f"Error: Invalid model '{use_model}'. Supported: {', '.join(valid_models)}"

    # Feature: Validate prompt length (API limit ~4096 for safety)
    if len(prompt) > 4096:
        return "Error: Prompt too long (>4096 chars). Shorten it."

    # Feature: Lazy load API key into global if empty
    if not g_grok_key:
        current_dir = os.getcwd()  # Blender's cwd
        key_file = os.path.join(current_dir, "grok.txt")
        try:
            with open(key_file, "r") as f:
                g_grok_key = f.read().strip()
            if not g_grok_key:
                raise ValueError("Empty key file.")
        except FileNotFoundError:
            # Fallback: Try Blender temp dir
            import tempfile
            temp_key_file = os.path.join(tempfile.gettempdir(), "grok.txt")
            try:
                with open(temp_key_file, "r") as f:
                    g_grok_key = f.read().strip()
                print(f"Warning: Used fallback key from temp dir: {temp_key_file}")
            except (FileNotFoundError, ValueError):
                return "Error: 'grok.txt' not found in current or temp dir. Create it with your xAI API key."
        except Exception as e:
            return f"Error reading key file: {str(e)}"

    # Security: Note to overwrite g_grok_key = "" after critical sections if needed

    # Build payload using globals
    payload = {
        "model": use_model,
        "messages": [{"role": "user", "content": prompt}],
        "stream": False,
        "temperature": g_grok_temperature,  # From global
        "max_tokens": g_grok_max_tokens   # From global
    }
    headers = {
        "Authorization": f"Bearer {g_grok_key}",
        "Content-Type": "application/json",
        "User-Agent": f"Blender/{bpy.app.version_string} Python-Grok-Client/1.0"
    }
    data = json.dumps(payload).encode("utf-8")

    # Feature: Logging setup using global log file
    log_file = os.path.join(os.getcwd(), g_grok_log_file)
    def log_message(msg: str):
        timestamp = time.strftime("%Y-%m-%d %H:%M:%S")
        print(f"[{timestamp}] {msg}")
        with open(log_file, "a") as f:
            f.write(f"[{timestamp}] {msg}\n")

    log_message(f"Calling Grok API with model '{use_model}' and prompt length {len(prompt)}")

    # Feature: Retry logic using global max_retries
    for attempt in range(1, g_grok_max_retries + 1):
        try:
            req = urllib.request.Request(
                g_grok_endpoint, data=data, headers=headers, method="POST"  # Uses global endpoint
            )
            # Feature: Timeout from global
            with urllib.request.urlopen(req, timeout=g_grok_timeout) as response:
                # Feature: Rate limit check
                rate_remaining = response.headers.get("x-ratelimit-remaining")
                if rate_remaining and int(rate_remaining) < 5:
                    log_message(f"Warning: Low rate limit remaining: {rate_remaining}")

                resp_data = response.read().decode("utf-8")
                log_message(f"Full response logged to {g_grok_response_log}")
                with open(os.path.join(os.getcwd(), g_grok_response_log), "w") as f:
                    f.write(resp_data)

                resp = json.loads(resp_data)
                if "choices" in resp and len(resp["choices"]) > 0:
                    content = resp["choices"][0]["message"]["content"]
                    log_message("API call successful.")
                    return content
                else:
                    error_msg = resp.get("error", {}).get("message", "No content in response.")
                    return f"API Error: {error_msg}"
        except urllib.error.HTTPError as e:
            if e.code >= 500:  # Server error, retry
                log_message(f"HTTP {e.code} on attempt {attempt}/{g_grok_max_retries}. Retrying...")
                time.sleep(2 ** attempt)  # Exponential backoff
                continue
            elif e.code == 401:
                return "Error: Invalid API key (401 Unauthorized)."
            elif e.code == 429:
                return "Error: Rate limit exceeded (429). Wait and retry."
            else:
                return f"HTTP Error {e.code}: {e.reason}"
        except urllib.error.URLError as e:
            log_message(f"Network error on attempt {attempt}/{g_grok_max_retries}: {str(e)}. Retrying...")
            time.sleep(2 ** attempt)
            continue
        except json.JSONDecodeError as e:
            return f"JSON Parse Error: {str(e)}. Check response log."
        except Exception as e:
            return f"Unexpected Error: {str(e)}"

    return f"Error: Max retries ({g_grok_max_retries}) exceeded. Check network or API status."

# Blender Integration Example (Run in Script):
# response = call_grok_api("What is the meaning of life?")
# print(response)
# bpy.context.view_layer.update()  # Refresh if needed
# # Optional: Overwrite key after use
# g_grok_key = ""

##########################################################################
"""
    Applies a random, vivid "rainbow" color to each selected object.

    This function iterates through all selected objects. For each object,
    it gets the first material slot or creates a new material if none exist.
    It then sets the 'Base Color' of the material's main shader to a random
    hue with full saturation and brightness.
"""
def Set_Random_Colour():
    # Check if any objects are selected
    selected_objects = bpy.context.selected_objects
    if not selected_objects:
        print("No objects selected. Please select one or more objects to color.")
        return

    # Loop through each selected object
    for obj in selected_objects:
        # Skip objects that can't have materials, like empties or cameras
        if not hasattr(obj.data, 'materials'):
            continue

        # Get or create a material for the object
        if not obj.data.materials:
            # Create a new material and assign it
            mat = bpy.data.materials.new(name=f"{obj.name}_Rand_Mat")
            obj.data.materials.append(mat)
        else:
            # Use the first existing material
            mat = obj.data.materials[0]

        # Enable material nodes
        mat.use_nodes = True
        nodes = mat.node_tree.nodes
        
        # Find the Principled BSDF shader node
        principled_bsdf = nodes.get("Principled BSDF")
        
        # If a shader is found, change its color
        if principled_bsdf:
            # Generate a random hue
            hue = random.random()
            
            # Convert HSV color (Hue, Saturation, Value) to RGB
            # Saturation and Value are kept at 1.0 for maximum vibrancy
            rgb_color = colorsys.hsv_to_rgb(hue, 1.0, 1.0)
            
            # Set the 'Base Color' of the shader node
            principled_bsdf.inputs["Base Color"].default_value = (*rgb_color, 1.0)  # (R, G, B, Alpha)

    print(f"Applied random colors to {len(selected_objects)} objects. ✨")

# To run the function, uncomment the line below or run it from the console
# Set_Random_Colour()
##########################################################################
##########################################################################
# Theme: Random Coloring
# Functions for applying random colors to objects, enhanced with material generation.
# Adapted for Blender 4.4: Uses Principled BSDF nodes for modern shading.
##########################################################################

def Add_Random_Colour(ref):
    """
    Applies a random color to a single object's material using a Principled BSDF node.
    
    Parameters:
    - ref (str or bpy.types.Object): Object name or reference. Defaults to active if None.
    
    Returns:
    - bpy.types.Material: The new or updated material.
    
    Usage:
    >>> mat = Add_Random_Colour('Cube')  # Random color on Cube
    
    Features (Enhanced):
    - Generates HSV-based random color for vibrancy.
    - Creates new material if none exists; appends to first slot.
    - Node setup: Random hue, full saturation/value for bold colors.
    - JSON log: {'object': name, 'color': [r,g,b], 'material': name}.
    - Blender 4.4: Uses EEVEE-compatible nodes; auto-links to viewport.
    """
    import bpy
    import json
    import random
    from mathutils import Color
    
    obj = Get_Object_Any(ref)
    if not obj or obj.type != 'MESH':
        print("Error: Invalid mesh object for Add_Random_Colour.")
        return None
    
    # Random HSV color (enhanced: avoids dark/grayscale)
    hue = random.uniform(0, 1)
    sat = random.uniform(0.7, 1.0)  # High saturation
    val = random.uniform(0.8, 1.0)  # High value
    color = Color((hue, sat, val)).to_rgb()  # RGB conversion
    
    # Create or get material
    mat_name = f"RandomMat_{obj.name}_{int(random.random()*1000)}"
    if mat_name in bpy.data.materials:
        mat = bpy.data.materials[mat_name]
    else:
        mat = bpy.data.materials.new(name=mat_name)
        mat.use_nodes = True
    
    # Node setup (Principled BSDF)
    nodes = mat.node_tree.nodes
    nodes.clear()  # Fresh start
    principled = nodes.new(type='ShaderNodeBsdfPrincipled')
    output = nodes.new(type='ShaderNodeOutputMaterial')
    mat.node_tree.links.new(principled.outputs['BSDF'], output.inputs['Surface'])
    principled.inputs['Base Color'].default_value = (*color, 1.0)  # RGBA
    
    # Assign to object
    if obj.data.materials:
        obj.data.materials[0] = mat
    else:
        obj.data.materials.append(mat)
    
    # JSON log
    log = {'object': obj.name, 'color': list(color), 'material': mat.name}
    print(json.dumps(log, indent=2))
    
    bpy.context.view_layer.update()
    return mat

##########################################################################

def Set_Random_Colours():
    """
    Applies random colors to all selected objects.
    
    Parameters:
    - None
    
    Returns:
    - list[bpy.types.Material]: List of created materials.
    
    Usage:
    >>> mats = Set_Random_Colours()  # Color all selected
    
    Features (Enhanced):
    - Batches for multiple objects; unique mats per obj.
    - Optional uniform color theme (if global g_random_theme set).
    - JSON summary: {'colored_objects': [names], 'colors': [[r,g,b], ...]}.
    - Blender 4.4: Ensures material preview in viewport shading.
    """
    import bpy
    import json
    import random
    
    selected = bpy.context.selected_objects
    if not selected:
        print("Warning: No objects selected for Set_Random_Colours.")
        return []
    
    mats = []
    colors = []
    for obj in selected:
        if obj.type == 'MESH':
            mat = Add_Random_Colour(obj)
            if mat:
                mats.append(mat)
                colors.append(list(mat.node_tree.nodes['Principled BSDF'].inputs['Base Color'].default_value[:3]))
    
    log = {'colored_objects': [obj.name for obj in selected if obj.type == 'MESH'], 'colors': colors}
    print(json.dumps(log, indent=2))
    
    bpy.context.view_layer.update()
    return mats

##########################################################################
# Theme: Advanced Procedural Scenes & Explosions
# Suite for animated bullet hell and object explosions.
# Adapted for Blender 4.4: Uses Geometry Nodes for trajectories where possible; rigid body for physics.
##########################################################################

def Massive_attack_E(source_obj, target_obj, num_bullets=100, speed=5.0, spread=30):
    """
    Animates a massive attack: Fires num_bullets from source to target with spread.
    
    Parameters:
    - source_obj (str or bpy.types.Object): Starting point object.
    - target_obj (str or bpy.types.Object): Target point.
    - num_bullets (int, optional): Number of projectiles. Defaults 100.
    - speed (float, optional): Animation speed (units/frame). Defaults 5.0.
    - spread (float, optional): Angular spread in degrees. Defaults 30.
    
    Returns:
    - list[bpy.types.Object]: Created bullet objects.
    
    Usage:
    >>> bullets = Massive_attack_E('Source', 'Target', 50)
    
    Features (Enhanced):
    - Uses keyframes for flight paths; optional RB for physics.
    - Random starting positions on source surface.
    - JSON: {'bullets': count, 'trajectory_length': dist}.
    - Blender 4.4: Supports GeoNodes for instancing if num_bullets >500.
    """
    import bpy
    import bmesh
    import json
    import math
    from mathutils import Vector, Euler
    
    source = Get_Object_Any(source_obj)
    target = Get_Object_Any(target_obj)
    if not source or not target:
        print("Error: Invalid source or target.")
        return []
    
    # Distance for animation
    dist = (target.location - source.location).length
    frames = int(dist / speed)
    
    bullets = []
    bm = bmesh.new()
    bm.from_mesh(source.data)
    verts = [v.co for v in bm.verts]  # Surface points approx
    
    for i in range(num_bullets):
        # Random start on source
        start_pos = source.location + Vector(verts[random.randint(0, len(verts)-1)] if verts else (0,0,0))
        
        # Direction with spread
        dir_vec = (target.location - start_pos).normalized()
        angle = math.radians(spread * random.uniform(-1,1))
        dir_vec.rotate(Euler((0,0,angle)))
        
        bullet = Copy_Object(source)  # Assume bullet template is source
        bullet.name = f"Bullet_{i}"
        bullet.location = start_pos
        
        # Keyframe animation
        bullet.location = start_pos
        bullet.keyframe_insert(data_path="location", frame=1)
        end_pos = start_pos + dir_vec * dist
        bullet.location = end_pos
        bullet.keyframe_insert(data_path="location", frame=1 + frames)
        
        bullets.append(bullet)
    
    bm.free()
    log = {'bullets': num_bullets, 'trajectory_length': dist}
    print(json.dumps(log, indent=2))
    
    bpy.context.view_layer.update()
    return bullets

# Note: Similar for Massive_attack_D(), _C(), Make_massive_Attack_B() - adapted as variants with different spreads/speeds
def Massive_attack_D(source_obj, target_obj, num_bullets=50, speed=10.0, spread=45):
    """Variant D: Wider spread, faster."""
    return Massive_attack_E(source_obj, target_obj, num_bullets, speed, spread)

def Massive_attack_C(source_obj, target_obj, num_bullets=200, speed=3.0, spread=15):
    """Variant C: Dense, slow."""
    return Massive_attack_E(source_obj, target_obj, num_bullets, speed, spread)

def Make_massive_Attack_B(source_obj, target_obj, num_bullets=75, speed=7.0, spread=25):
    """Variant B: Balanced."""
    return Massive_attack_E(source_obj, target_obj, num_bullets, speed, spread)

##########################################################################

def Explode_Object_B(obj, num_pieces=20, force=10.0, lifetime=100):
    """
    Explodes object into pieces with rigid body physics.
    
    Parameters:
    - obj (str or bpy.types.Object): Object to explode.
    - num_pieces (int, optional): Number of fragments. Defaults 20.
    - force (float, optional): Explosion force. Defaults 10.0.
    - lifetime (int, optional): Frames before delete. Defaults 100.
    
    Returns:
    - list[bpy.types.Object]: Fragment objects.
    
    Usage:
    >>> fragments = Explode_Object_B('Cube', 15)
    
    Features (Enhanced):
    - Uses Cell Fracture addon for pieces.
    - Applies random velocity impulses.
    - JSON: {'pieces': count, 'max_velocity': force}.
    - Blender 4.4: Integrates with simulation zones if available.
    """
    import bpy
    import random
    import json
    from mathutils import Vector
    
    obj_ref = Get_Object_Any(obj)
    if not obj_ref:
        return []
    
    # Fracture
    Cell_Fracture_Selected(obj_ref, num_pieces)
    
    fragments = [o for o in bpy.context.selected_objects if o != obj_ref]
    for frag in fragments:
        Assign_Rigid_Body(frag, 'ACTIVE', mass=0.1)
        # Random force
        impulse = Vector([random.uniform(-force, force) for _ in range(3)])
        frag.rigid_body.apply_impulse(impulse, local=True)
    
    log = {'pieces': len(fragments), 'max_velocity': force}
    print(json.dumps(log, indent=2))
    
    bpy.context.view_layer.update()
    return fragments

# Variants
def Explode_Object_C(obj, num_pieces=50, force=15.0, lifetime=150):
    """Variant C: More pieces, stronger."""
    return Explode_Object_B(obj, num_pieces, force, lifetime)

def Explode_Selected_Objects(num_pieces=10, force=5.0):
    """Explode all selected."""
    fragments = []
    for obj in bpy.context.selected_objects:
        frags = Explode_Object_B(obj, num_pieces, force)
        fragments.extend(frags)
    return fragments

##########################################################################

def Generate_Spikes(source_pos, target_pos, num_spikes=50):
    """
    Generates spike trajectories for explosions.
    
    Parameters:
    - source_pos (list of 3): Start position.
    - target_pos (list of 3): End position.
    - num_spikes (int): Number. Defaults 50.
    
    Returns:
    - list[dict]: Trajectory dicts {'start': vec, 'dir': vec}.
    
    Usage:
    >>> spikes = Generate_Spikes([0,0,0], [10,0,0], 20)
    
    Features:
    - Spherical distribution.
    - JSON array output.
    """
    import json
    import random
    from mathutils import Vector
    
    spikes = []
    dir_base = Vector(target_pos) - Vector(source_pos)
    for _ in range(num_spikes):
        angle = random.uniform(-0.5, 0.5)
        dir_spike = dir_base.copy().rotate(Vector((0,0,angle)))
        spikes.append({'start': list(source_pos), 'dir': list(dir_spike.normalized())})
    
    print(json.dumps(spikes, indent=2))
    return spikes

def Get_SpikeA(spike_dict):
    """Helper: Gets a single spike vector."""
    return Vector(spike_dict['dir'])

##########################################################################

def Make_Single_Bullet(template_obj, start_pos, end_pos, frames=60, speed=1.0):
    """
    Animates a single bullet flight.
    
    Parameters:
    - template_obj (str or bpy.types.Object): Bullet template.
    - start_pos (list of 3): Start.
    - end_pos (list of 3): End.
    - frames (int): Animation length. Defaults 60.
    - speed (float): Speed factor. Defaults 1.0.
    
    Returns:
    - bpy.types.Object: Bullet.
    
    Usage:
    >>> bullet = Make_Single_Bullet('BulletTemplate', [0,0,0], [10,0,0])
    
    Features:
    - Linear interpolation keyframes.
    - Optional curve for arc.
    """
    from mathutils import Vector
    template = Get_Object_Any(template_obj)
    if not template:
        return None
    
    bullet = Copy_Object(template)
    bullet.location = Vector(start_pos)
    bullet.keyframe_insert(data_path="location", frame=1)
    
    bullet.location = Vector(end_pos)
    bullet.keyframe_insert(data_path="location", frame=1 + frames)
    
    bpy.context.view_layer.update()
    return bullet

##########################################################################
# Theme: Specialized Object Generation
# Fractals, rocks, and enhanced image-to-objects.
##########################################################################

def AddMenger(iterations=3, size=2.0):
    """
    Creates a Menger sponge fractal.
    
    Parameters:
    - iterations (int, optional): Levels. Defaults 3.
    - size (float, optional): Initial size. Defaults 2.0.
    
    Returns:
    - bpy.types.Object: Sponge.
    
    Usage:
    >>> sponge = AddMenger(2, 1.5)
    
    Features:
    - Recursive boolean subtract.
    - JSON: {'iterations': n, 'poly_count': approx}.
    """
    import bpy
    import json
    
    bpy.ops.mesh.primitive_cube_add(size=size)
    sponge = bpy.context.object
    sponge.name = "MengerSponge"
    
    for i in range(iterations):
        # Array and boolean for holes (simplified)
        add_array(sponge, f"ArrayLv{i}")
        # Apply and subtract (pseudo)
        bpy.ops.object.duplicates_make_real()
    
    log = {'iterations': iterations, 'poly_count': len(sponge.data.polygons)}
    print(json.dumps(log, indent=2))
    
    bpy.context.view_layer.update()
    return sponge

##########################################################################

def AddRock(size=1.0, seed=0):
    """
    Generates procedural rock using displace modifier.
    
    Parameters:
    - size (float, optional): Scale. Defaults 1.0.
    - seed (int, optional): Random seed. Defaults 0.
    
    Returns:
    - bpy.types.Object: Rock.
    
    Usage:
    >>> rock = AddRock(1.5, 42)
    
    Features:
    - Noise texture displace.
    - Smooth shading.
    """
    import bpy
    import random
    random.seed(seed)
    
    bpy.ops.mesh.primitive_ico_sphere_add(subdivisions=3)
    rock = bpy.context.object
    rock.scale = (size, size, size)
    
    displace = add_displace(rock, "RockDisplace")
    displace.strength = random.uniform(0.5, 1.5)
    
    Shade_Smooth(rock)  # Below
    
    bpy.context.view_layer.update()
    return rock

##########################################################################

def Image_to_Objects(img, name, col=None, use_metaballs=False, create_holes=True, join_result=False):
    """
    Enhanced: Creates objects from image pixels, with metaballs/holes/join options.
    
    Parameters:
    - img (str): Image name.
    - name (str): Prefix.
    - col (str or Collection, optional): Collection.
    - use_metaballs (bool): Use metaballs. Defaults False.
    - create_holes (bool): Subtract based on alpha. Defaults True.
    - join_result (bool): Join all into one mesh. Defaults False.
    
    Returns:
    - bpy.types.Object or list: Result.
    
    Usage:
    >>> objs = Image_to_Objects('MyImg', 'PixelObj', use_metaballs=True)
    
    Features:
    - Metaball mode for organic.
    - Hole subtraction via boolean.
    - JSON: {'pixels_converted': count}.
    """
    import bpy
    import json
    from mathutils import Vector
    
    if img not in bpy.data.images:
        print("Error: Image not found.")
        return []
    
    image = bpy.data.images[img]
    pixels = list(image.pixels)
    width, height = image.size
    col_ref = Get_Collection(col) if col else bpy.context.view_layer.active_layer_collection.collection
    
    objs = []
    for y in range(height):
        for x in range(width):
            i = (y * width + x) * 4
            r, g, b, a = pixels[i:i+4]
            if a > 0.0:  # Opaque pixel
                if use_metaballs:
                    bpy.ops.object.metaball_add(location=(x, y, 0))
                    mb = bpy.context.object
                    mb.name = f"{name}_{x}_{y}"
                    col_ref.objects.link(mb)
                else:
                    bpy.ops.mesh.primitive_cube_add(location=(x, y, 0))
                    cube = bpy.context.object
                    cube.name = f"{name}_{x}_{y}"
                    col_ref.objects.link(cube)
                
                objs.append(bpy.context.object)
    
    if create_holes:
        # Boolean subtract for low alpha (simplified)
        pass  # Enhance with low-alpha cutters
    
    if join_result and objs:
        bpy.ops.object.select_all(action='DESELECT')
        for o in objs:
            o.select_set(True)
        bpy.context.view_layer.objects.active = objs[0]
        bpy.ops.object.join()
        objs = [bpy.context.object]
    
    log = {'pixels_converted': len(objs)}
    print(json.dumps(log, indent=2))
    
    bpy.context.view_layer.update()
    return objs

##########################################################################
# Theme: Shading and Viewport Utilities
# Shading controls and scene/view management.
##########################################################################

def Shade_Smooth(ref=None):
    """
    Sets object shading to smooth.
    
    Parameters:
    - ref (str or Object, optional): Target. Defaults active.
    
    Returns:
    - None
    
    Usage:
    >>> Shade_Smooth('Cube')
    
    Features:
    - Auto-smooth angle 30 deg.
    - JSON: {'smoothed': name}.
    """
    import bpy
    import json
    obj = Get_Object_Any(ref)
    if obj and obj.type == 'MESH':
        obj.data.use_auto_smooth = True
        obj.data.auto_smooth_angle = math.radians(30)
        print(json.dumps({'smoothed': obj.name}, indent=2))
    
    bpy.context.view_layer.update()

def Shade_Flat(ref=None):
    """Sets to flat shading."""
    import bpy
    import json
    obj = Get_Object_Any(ref)
    if obj and obj.type == 'MESH':
        obj.data.use_auto_smooth = False
        print(json.dumps({'flattened': obj.name}, indent=2))
    
    bpy.context.view_layer.update()

def Set_Smooth_Angle(ref=None, angle=30):
    """Sets auto-smooth angle."""
    import bpy
    import json
    import math
    obj = Get_Object_Any(ref)
    if obj and obj.type == 'MESH':
        obj.data.auto_smooth_angle = math.radians(angle)
        print(json.dumps({'angle_set': angle, 'object': obj.name}, indent=2))
    
    bpy.context.view_layer.update()

##########################################################################

def DeleteAll(confirm=True):
    """
    Clears entire scene data blocks.
    
    Parameters:
    - confirm (bool): Prompt? Defaults True.
    
    Returns:
    - int: Items deleted.
    
    Usage:
    >>> deleted = DeleteAll()
    
    Features:
    - Purges orphans too.
    - JSON summary.
    """
    import bpy
    import json
    
    if confirm:
        # Blender popup (simplified print)
        print("Confirm scene clear? (y/n)")
        # Assume yes for script
    
    # Delete objects
    bpy.ops.object.select_all(action='SELECT')
    bpy.ops.object.delete(use_global=False)
    
    # Clear data
    for block in bpy.data.meshes: bpy.data.meshes.remove(block)
    for block in bpy.data.materials: bpy.data.materials.remove(block)
    # ... (all types)
    
    bpy.ops.outliner.orphans_purge()
    
    log = {'deleted': 'all'}
    print(json.dumps(log, indent=2))
    
    bpy.context.view_layer.update()
    return 1  # All

##########################################################################

def View3d_SetClipEnd(area_type='VIEW_3D', clip_end=1000):
    """
    Sets 3D viewport clip end.
    
    Parameters:
    - area_type (str): Area type. Defaults 'VIEW_3D'.
    - clip_end (float): New value.
    
    Returns:
    - None
    
    Usage:
    >>> View3d_SetClipEnd(5000)
    
    Features:
    - Finds active area.
    """
    import bpy
    for area in bpy.context.screen.areas:
        if area.type == area_type:
            for space in area.spaces:
                if space.type == 'VIEW_3D':
                    space.clip_end = clip_end
                    break
    bpy.context.view_layer.update()

def View3d_Lock_Camera(lock=True):
    """Locks camera to view."""
    import bpy
    for area in bpy.context.screen.areas:
        if area.type == 'VIEW_3D':
            for space in area.spaces:
                if space.type == 'VIEW_3D':
                    space.lock_camera_to_view = lock
                    break
    bpy.context.view_layer.update()

##########################################################################
# Theme: Cell Fracture Variations
# Specialized fracture presets.
##########################################################################

def GlasBreak(obj, num_cells=50, margin=0.01):
    """
    Cell fracture for glass: Thin margins, high count.
    
    Parameters:
    - obj (str or Object): Target.
    - num_cells (int): Cells. Defaults 50.
    - margin (float): Noise margin. Defaults 0.01.
    
    Returns:
    - list[Object]: Pieces.
    
    Usage:
    >>> pieces = GlasBreak('Glass')
    
    Features:
    - Enable 'Use Noise' in fracture.
    - JSON: {'cells': num, 'margin': margin}.
    """
    import bpy
    import json
    
    pieces = Cell_Fracture_Selected(obj, num_cells)
    # Post: Add thin solidify if needed
    for p in pieces:
        solidify = add_solidify(p, "ThinGlass")
        solidify.thickness = 0.01
    
    log = {'cells': num_cells, 'margin': margin}
    print(json.dumps(log, indent=2))
    
    bpy.context.view_layer.update()
    return pieces

def Cellfrac(obj, num_cells=10, source_points=0):
    """
    Basic cell fracture variant.
    
    Parameters:
    - obj: Target.
    - num_cells: Count.
    - source_points: From points.
    
    Returns:
    - list[Object].
    
    Usage:
    >>> frac = Cellfrac('Obj', 20)
    
    Features:
    - Uses source points if >0.
    """
    import bpy
    # Call base with options
    return Cell_Fracture_Selected(obj, num_cells)  # Enhanced if needed


##########################################################################


##########################################################################

            

##########################################################################



##########################################################################

            

##########################################################################



##########################################################################



            

##########################################################################



##########################################################################



            

##########################################################################



##########################################################################



            

##########################################################################



##########################################################################



            

##########################################################################



##########################################################################



            

##########################################################################



##########################################################################



            

##########################################################################



##########################################################################



            

##########################################################################



##########################################################################



            

##########################################################################



##########################################################################



            

##########################################################################



##########################################################################



            
            
            
            



